<?php
/**
* Plugin Name: Platon CDN
* Plugin URI: https://platon.org/
* Description: Use Platon CDN for media.
* Version: 1.29
* Author: Platon Technologies
* Author URI: https://platon.org/
* Update URI: https://wordpress.platon.sk/platon-cdn/
* License: GPL2
*/

require_once __DIR__.'/constants.php';
require_once __DIR__.'/lib/PlatonCDN.php';

class Platon_CDN_plugin extends PlatonCDN {

	public static $version = '1.29';
	public static $table_cdn = 'platon_cdn';

	public function __construct() {
		parent::__construct();
		global $wpdb;
		// Check modificaions in database
		$this->dbCheck();
		// Load action for modifying URL s and uploading media
        $this->registerHooks();
	}

	public static function menu() {
		add_submenu_page(
			'tools.php',
			'Platon CDN',
			'Platon CDN',
			'moderate_comments', //'manage_options',
			plugin_dir_path(__FILE__).'admin/view.php'
		);
	}

	public function admin_init() {
		// specialny pripad, kedze inicializujeme plugin "stale" tak nemame nacitanu konfiguraciu este, nacitavame dodatocne
		$this->checkWordpressConfig();
		// Do actions for admin UI
		$this->doActions();
	}

	/**
	 * ACTIVATION OF PLUGIN
	 */
	public function onActivate() {
		$this->dbInit();
	}

	/**
	 * DEACTIVATION OF PLUGIN
	 */
	public function onDeactivate() {
		$this->connectionSelected(0);
	}

	/**
	 * PLUGIN UPDATE
	 */
	// https://rudrastyh.com/wordpress/self-hosted-plugin-update.html
	public function pluginsAPI($res, $action, $args) {
		// do nothing if this is not about getting plugin information
		if( 'plugin_information' !== $action ) {
			return $res;
		}

		// do nothing if it is not our plugin
		if( plugin_basename( __DIR__ ) !== $args->slug ) {
			return $res;
		}

		// info.json is the file with the actual plugin information on your server
		$remote = wp_remote_get(
			'https://wordpress.platon.sk/platon-cdn/info.json',
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json'
				)
			)
		);

		// do nothing if we don't get the correct response from the server
		if (is_wp_error( $remote )
			|| 200 !== wp_remote_retrieve_response_code( $remote )
			|| empty( wp_remote_retrieve_body( $remote )))
		{
			return $res;
		}

		$remote = json_decode( wp_remote_retrieve_body( $remote ) );

		$res = new stdClass();
		$res->name = $remote->name;
		$res->slug = $remote->slug;
		$res->author = $remote->author;
		$res->author_profile = $remote->author_profile;
		$res->version = $remote->version;
		$res->tested = $remote->tested;
		$res->requires = $remote->requires;
		$res->requires_php = $remote->requires_php;
		$res->download_link = $remote->download_url;
		$res->trunk = $remote->download_url;
		$res->last_updated = $remote->last_updated;
		$res->sections = array(
			'description' => $remote->sections->description,
			'installation' => $remote->sections->installation,
			'changelog' => $remote->sections->changelog
			// you can add your custom sections (tabs) here
		);
		// in case you want the screenshots tab, use the following HTML format for its content:
		// <ol><li><a href="IMG_URL" target="_blank"><img src="IMG_URL" alt="CAPTION" /></a><p>CAPTION</p></li></ol>
		if( ! empty( $remote->sections->screenshots ) ) {
			$res->sections[ 'screenshots' ] = $remote->sections->screenshots;
		}

		$res->banners = array(
			'low' => $remote->banners->low,
			'high' => $remote->banners->high
		);

		return $res;
	}

	public function updatePlugin( $transient ) {
		//print_r($transient);
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$remote = wp_remote_get(
			'https://wordpress.platon.sk/platon-cdn/info.json',
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json'
				)
			)
		);

		if (is_wp_error( $remote )
			|| 200 !== wp_remote_retrieve_response_code( $remote )
			|| empty( wp_remote_retrieve_body( $remote )))
		{
			return $transient;
		}

		$remote = json_decode( wp_remote_retrieve_body( $remote ) );

		// your installed plugin version should be on the line below! You can obtain it dynamically of course
		if(
			$remote
			&& version_compare( Platon_CDN_plugin::$version, $remote->version, '<' )
			&& version_compare( $remote->requires, get_bloginfo( 'version' ), '<=' )
			&& version_compare( $remote->requires_php, PHP_VERSION, '<=' )
		) {

			$res = new stdClass();
			$res->slug = $remote->slug;
			$res->plugin = plugin_basename( __FILE__ ); // it could be just YOUR_PLUGIN_SLUG.php if your plugin doesn't have its own directory
			$res->new_version = $remote->version;
			$res->tested = $remote->tested;
			$res->package = $remote->download_url;
			$transient->response[ $res->plugin ] = $res;

			//$transient->checked[$res->plugin] = $remote->version;
		}

		return $transient;
	}

	/**
	 * ACTIONS FOR ADMIN UI
	 */
	public function doActions() {
		// Connections
		if (isset($_REQUEST['save'])) {
			if (isset($_REQUEST['cdn_id'])) {
				$this->connectionUpdate($_REQUEST['cdn_id'], $_REQUEST['data']);
			} else {
				$this->connectionCreate($_REQUEST['data']);
			}
		} else if (isset($_REQUEST['connection-delete'])) {
			$this->connectionDelete($_REQUEST['cdn_id']);
		} else if (isset($_REQUEST['connection-select'])) {
			$this->connectionSelected($_REQUEST['cdn_id']);
		} else if (isset($_REQUEST['connection-disable'])) {
			$this->connectionSelected(0);
		}
		// General settings
		else if (isset($_REQUEST['conversion'])) {
			$this->conversionEnabled($_REQUEST['conversion']);
		}
		else if (isset($_REQUEST['conversion-quality-update'])) {
			$this->conversionQuality($_REQUEST['data']['conversion_quality']);
		}
		// Prefixes
		else if (isset($_REQUEST['prefix-create'])) {
			$this->prefixCreate($_REQUEST['data']);
		} else if (isset($_REQUEST['prefix-update'])) {
			$this->prefixUpdate($_REQUEST['prefix_id'], $_REQUEST['data']);
		} else if (isset($_REQUEST['prefix-delete'])) {
			$this->prefixDelete($_REQUEST['prefix_id']);
		}
		// Logs Explorer
		else if (isset($_REQUEST['logpath-save'])) {
			$this->logPath($_REQUEST['logpath']);
			$_REQUEST['logs-explorer'] = true;
		} else if (isset($_REQUEST['logpath-fetch'])) {
			echo json_encode($this->logFetch($_REQUEST['filename']));
			exit(0);
		}
	}

	/**
	 * URL REWRITING
	 */
	public function registerHooks()
    {
		/** URL RE WRITING HOOKS */
		add_filter('wp_get_attachment_url', array($this, 'wp_get_attachment_url'), 99, 2);
		// add_filter('wp_get_attachment_image_attributes', array($this, 'wp_get_attachment_image_attributes'), 99, 3);
		add_filter('wp_calculate_image_srcset', array($this, 'wp_calculate_image_srcset'), 99, 5);
		add_filter('get_attached_file', array($this, 'get_attached_file'), 10, 2);
		add_filter('wp_get_original_image_path', array($this, 'get_attached_file'), 10, 2);

		/** FILE MANAGEMENT HOOKS */
		// add_filter('wp_unique_filename', array($this, 'wp_unique_filename'), 10, 3);
		// add_filter('wp_update_attachment_metadata', array($this, 'update_attachment_metadata'), 110, 2);
		// add_filter('wp_generate_attachment_metadata', array($this, 'wp_generate_attachment_metadata'), 110, 3);
		// add_filter('delete_attachment', array($this, 'delete_attachment'), 20);
		// add_filter('update_attached_file', array($this, 'update_attached_file'), 100, 2);
		// add_filter('load_image_to_edit_path', array($this, 'load_image_to_edit_path'), 10, 3 );
	}

	private function extractLastURL($url) {
		$url_lower = strtolower($url);
		$pos_https = strrpos($url_lower, 'https://');
		$pos_http = strrpos($url_lower, 'http://');
		if ($pos_https === false
			&& $pos_http === false)
		{
			return $url;
		}
		$pos = max(intval($pos_https), intval($pos_http));
		$return_url = substr($url, $pos);
		return $return_url;
	}

	// Get attachment url
	// https://developer.wordpress.org/reference/functions/wp_get_attachment_url/
	// https://developer.wordpress.org/reference/hooks/wp_get_attachment_url/
	public function wp_get_attachment_url($url, $attachment_id)	{
		$ret = $this->extractLastURL($url);
		return $ret;
	}


	// Filters the list of attachment image attributes.
	// https://developer.wordpress.org/reference/hooks/wp_get_attachment_image_attributes/
	public function wp_get_attachment_image_attributes($attr, $attachment, $size='thumbnail') {
		return $attr;
	}

	// Change src attributes.
	// https://developer.wordpress.org/reference/hooks/wp_calculate_image_srcset/
	public function wp_calculate_image_srcset($sources, $size_array, $image_src, $attachment_meta, $attachment_id = 0) {
		$remote_url_basic = dirname($image_src).'/';
		if (is_array($sources)) foreach ($sources as $size => $params) {
			$image_filename = basename($params['url']);
			$sources[$size]['url'] = $remote_url_basic.$image_filename;
		}
		return $sources;
	}

	// Return the provider URL when the local file is missing
	// unless we know who the calling process is and we are happy
	// to copy the file back to the server to be used.
	// https://developer.wordpress.org/reference/hooks/get_attached_file/
	// https://developer.wordpress.org/reference/hooks/wp_get_original_image_path/
	public function get_attached_file($file, $attachment_id) {
		$ret = $this->extractLastURL($file);
		return $ret;
	}

	// Create unique names for files effects mainly on delete files from server settings
	// https://developer.wordpress.org/reference/hooks/wp_unique_filename/
	public function wp_unique_filename($filename, $ext, $dir) {
		return $filename;
	}

	// Function to execute on wp_update_attachment_metadata
	// https://developer.wordpress.org/reference/hooks/wp_update_attachment_metadata/
	public function update_attachment_metadata($attachment_meta, $attachment_id) {
		return $attachment_meta;
	}

	// Function to execute on wp_generate_attachment_metadata
	// To delete files from server only after updating all meta
	// https://developer.wordpress.org/reference/hooks/wp_generate_attachment_metadata/
	 public function wp_generate_attachment_metadata( $attachment_meta, $attachment_id, $action) {
		return $attachment_meta;
	 }

	// Function to remove data from provider by attachment ID
	// https://developer.wordpress.org/reference/hooks/delete_attachment/
	public function delete_attachment($post_id) {
		return $post_id;
	}

	// Allow processes to update the file on provider via update_attached_file()
	// https://developer.wordpress.org/reference/hooks/update_attached_file/
	public function update_attached_file($file, $attachment_id) {
		return $file;
	}

	// Function to execute on load_image_to_edit_path
	// https://developer.wordpress.org/reference/hooks/load_image_to_edit_path/
	public function load_image_to_edit_path($path, $attachment_id, $size) {
		return $path;
	}

}

/**
 * INITIALIZATION OF PLUGIN
 */
function platon_cdn_init() {
	$GLOBALS['platon_cdn_plugin'] = new Platon_CDN_plugin();
}

//add_action('admin_init', 'platon_cdn_init' );
platon_cdn_init(); // initialize always

add_action('admin_menu', array($GLOBALS['platon_cdn_plugin'], 'menu'));
add_action('admin_init', array($GLOBALS['platon_cdn_plugin'], 'admin_init'));
register_activation_hook(__FILE__, array($GLOBALS['platon_cdn_plugin'], 'onActivate'));
register_deactivation_hook( __FILE__, array($GLOBALS['platon_cdn_plugin'],'onDeactivate') );
add_filter( 'plugins_api', array($GLOBALS['platon_cdn_plugin'], 'pluginsAPI'), 20, 3);
add_filter( 'site_transient_update_plugins', array($GLOBALS['platon_cdn_plugin'], 'updatePlugin') );



?>
