<?php
/**
 * Developed by Igor Mino <mino@platon.net>
 * Copyright (c) 2024 Platon Technologies, s.r.o.
 * All rights reserved.
*/

require_once __DIR__.'/../constants.php';

class PlatonCDN {

	public static $table_cdn = null;
	public static $table_prefixes = null;
	public static $encryption = array(
		'method' => 'AES-256-CBC',
		'key' => 'w2fP80cXBXyCcr69A5lQuxFeIaWlIVhB', // random generated
		'options' => OPENSSL_RAW_DATA,
		'iv' => '0390201749348909' // random generated
	);

	public function __construct() {
		global $wpdb;
		// Check worpdress configuration
		$this->checkWordpressConfig();
		PlatonCDN::$table_cdn = $wpdb->prefix.'platon_cdn';
		PlatonCDN::$table_prefixes = $wpdb->prefix.'platon_cdn_prefixes';
	}

	protected function checkWordpressConfig() {
		if (defined('PLATON_CDN_ENCRYPTION_KEY')) {
			PlatonCDN::$encryption['key'] = PLATON_CDN_ENCRYPTION_KEY;
		}
		if (defined('PLATON_CDN_ENCRYPTION_IV')) {
			PlatonCDN::$encryption['iv'] = PLATON_CDN_ENCRYPTION_IV;
		}
	}

	/**
	 * DB CHECK
	 */
	public function dbInit() {
		global $wpdb;
		$query = 'SHOW TABLES LIKE `'.PlatonCDN::$table_cdn.'`';
		$table = $wpdb->get_row($query, ARRAY_A);
		if (is_null($table)) {
			$query = 'CREATE TABLE `'.PlatonCDN::$table_cdn.'` (
					`cdn_id` INT(11) PRIMARY KEY AUTO_INCREMENT,
					`connection_type` ENUM("FTP", "SFTP", "API") DEFAULT NULL,
					`hostname` VARCHAR(255) DEFAULT NULL,
					`port` INT(11) NOT NULL DEFAULT "22",
					`username` VARCHAR(255) DEFAULT NULL,
					`password` VARCHAR(255) DEFAULT NULL,
					`upload_dir` VARCHAR(255) DEFAULT NULL,
					`upload_url` VARCHAR(255) DEFAULT NULL
				)';
			$wpdb->query($query);
		}
	}

	public function dbCheck() {
		global $wpdb;
		$db_version = get_option(PLATON_CDN_DB_VERSION, 0);
		if ($db_version <= 0) {
			$query = 'SHOW TABLES LIKE `'.PlatonCDN::$table_prefixes.'`';
			$table = $wpdb->get_row($query, ARRAY_A);
			if (is_null($table)) {
				$query = 'CREATE TABLE `'.PlatonCDN::$table_prefixes.'` (
						`prefix_id` INT(11) PRIMARY KEY AUTO_INCREMENT,
						`content` VARCHAR(255) DEFAULT NULL
					)';
				$wpdb->query($query);
			}
			update_option(PLATON_CDN_DB_VERSION, 1);
		}
		/* example for update
		if ($db_version <= 1) {
			$query = 'DESCRIBE  "'.PlatonCDN::$table_cdn.'" "result"';
			$col_check = $wpdb->get_row($query, ARRAY_A); //echo 'col_check='; print_r($col_check);
			if (is_null($col_check)) {
				$fix_query = 'ALTER TABLE `'.$wpdb->prefix.'platon_cdn` ADD `result` VARCHAR(255) DEFAULT NULL AFTER `command`';
				$wpdb->query($fix_query);
			}
			update_option(PLATON_CDN_DB_VERSION, 2);
		}
		*/
	}

	/**
	 * HELPER FUNCTIONS
	 */
	public function basicEncrypt($data) {
		if (strlen($data) <= 0) return '';
		$encryptedData = openssl_encrypt($data,
			PlatonCDN::$encryption['method'],
			PlatonCDN::$encryption['key'],
			PlatonCDN::$encryption['options'],
			PlatonCDN::$encryption['iv']);
		$encryptedData_base64 = base64_encode($encryptedData);
		return $encryptedData_base64;
	}

	public function basicDecrypt($data) {
		if (strlen($data) <= 0) return '';
		$data_frombase64 = base64_decode($data);
		$decryptedData = openssl_decrypt($data_frombase64,
			PlatonCDN::$encryption['method'],
			PlatonCDN::$encryption['key'],
			PlatonCDN::$encryption['options'],
			PlatonCDN::$encryption['iv']);
		return $decryptedData;
	}

	public function webpImage($source, $quality = 100) {
		if (!file_exists($source)) {
			return false;
		}
		if (!extension_loaded('gd')) {
			return false;
		}
		$info = getimagesize($source);
		$isAlpha = false;
		if ($info['mime'] == 'image/jpeg')
			$image = imagecreatefromjpeg($source);
		elseif ($isAlpha = $info['mime'] == 'image/gif') {
			$image = imagecreatefromgif($source);
		} elseif ($isAlpha = $info['mime'] == 'image/png') {
			$image = imagecreatefrompng($source);
		} else {
			return false;
		}
		if ($isAlpha) {
			imagepalettetotruecolor($image);
			imagealphablending($image, true);
			imagesavealpha($image, true);
		}
		$stream = fopen('php://memory', 'w+');
		imagewebp($image, $stream, $quality);
		imagedestroy($image);
		rewind($stream);
		$ret = stream_get_contents($stream);
		fclose($stream);
		return $ret;
	}

	/**
	 * CONNECTIONS
	 */
	public function connectionSelected($cdn_id = null) {
		if (is_null($cdn_id)) {
			return get_option(PLATON_CDN_CONNECTION_ID, 0);
		} else {
			return update_option(PLATON_CDN_CONNECTION_ID, $cdn_id);
		}
	}

	public function connections() {
		global $wpdb;
		$query = sprintf('SELECT * FROM %s ORDER BY cdn_id ASC', PlatonCDN::$table_cdn);
		$all = $wpdb->get_results($query, ARRAY_A);
		if (is_array($all)) foreach ($all as $key => $row) {
			$row['password'] = $this->basicDecrypt($row['password']);
		}
		return $all;
	}

	public function connectionDefault() {
		return array(
			'connection_type' => '',
			'hostname' => '',
			'port' => '22',
			'username' => '',
			'password' => '',
			'upload_dir' => '',
			'upload_url' => '',
		);
	}

	public function connectionGet($cdn_id) {
		global $wpdb;
		$query = sprintf('SELECT * FROM %s  WHERE cdn_id = %d', PlatonCDN::$table_cdn, intval($cdn_id));
		$one = $wpdb->get_row($query, ARRAY_A);
		$one['password'] = $this->basicDecrypt($one['password']);
		return $one;
	}

	public function connectionTypes() {
		return array(
			'FTP' => 'FTP',
			'SFTP' => 'SFTP',
			'API' => 'API'
		);
	}

	public function connectionCreate($data) {
		global $wpdb;
		if (strlen($data['connection_type']) <= 0
			|| strlen($data['hostname']) <= 0
			|| strlen($data['username']) <= 0
			|| strlen($data['password']) <= 0)
		{
			return false;
		}
		$query = sprintf('INSERT INTO %s'
			.' (connection_type, hostname, port, username, password, upload_dir, upload_url)'
			.' VALUES ("%s", "%s", %d, "%s", "%s", "%s", "%s")',
			PlatonCDN::$table_cdn,
			$data['connection_type'],
			$data['hostname'],
			intval($data['port']),
			$data['username'],
			$this->basicEncrypt($data['password']),
			$data['upload_dir'],
			$data['upload_url']);
		return $wpdb->query($query);
	}

	public function connectionUpdate($cdn_id, $data) {
		global $wpdb;
		$query = sprintf('UPDATE %s SET'
			.' connection_type = "%s",'
			.' hostname = "%s",'
			.' port = %d,'
			.' username = "%s",'
			.' password = "%s",'
			.' upload_dir = "%s",'
			.' upload_url = "%s"'
			.' WHERE cdn_id = %d',
			PlatonCDN::$table_cdn,
			$data['connection_type'],
			$data['hostname'],
			intval($data['port']),
			$data['username'],
			$this->basicEncrypt($data['password']),
			$data['upload_dir'],
			$data['upload_url'],
			intval($cdn_id));
		return $wpdb->query($query);
	}

	public function connectionDelete($cdn_id) {
		global $wpdb;
		$query = sprintf('DELETE FROM %s WHERE cdn_id = %d', PlatonCDN::$table_cdn, intval($cdn_id));
		return $wpdb->query($query);
	}

	/**
	 * GENERAL SETTINGS
	 */
	public function conversionEnabled($enabled = null) {
		if (is_null($enabled)) {
			return get_option(PLATON_CDN_CONVERSION_ENABLED, 0) == 1;
		} else {
			return update_option(PLATON_CDN_CONVERSION_ENABLED, $enabled ? 1 : 0);
		}
	}

	public function conversionQuality($quality = null) {
		if (is_null($quality)) {
			return get_option(PLATON_CDN_CONVERSION_QUALITY, 80);
		} else {
			$quality = intval($quality);
			$quality = min(100, max(0, $quality));
			return update_option(PLATON_CDN_CONVERSION_QUALITY, $quality);
		}
	}

	/**
	 * PREFIXES
	 */
	public function prefixes() {
		global $wpdb;
		$query = sprintf('SELECT * FROM %s ORDER BY prefix_id ASC', PlatonCDN::$table_prefixes);
		$all = $wpdb->get_results($query, ARRAY_A);
		return $all;
	}

	public function prefixCreate($data) {
		global $wpdb;
		if (strlen($data['content']) <= 0)
		{
			return false;
		}
		$query = sprintf('INSERT INTO %s'
			.' (content)'
			.' VALUES ("%s")',
			PlatonCDN::$table_prefixes,
			$data['content']);
		return $wpdb->query($query);
	}

	public function prefixUpdate($prefix_id, $data) {
		global $wpdb;
		$query = sprintf('UPDATE %s SET'
			.' content = "%s"'
			.' WHERE prefix_id = %d',
			PlatonCDN::$table_prefixes,
			$data['content'],
			intval($prefix_id));
		return $wpdb->query($query);
	}

	public function prefixDelete($prefix_id) {
		global $wpdb;
		$query = sprintf('DELETE FROM %s WHERE prefix_id = %d', PlatonCDN::$table_prefixes, intval($prefix_id));
		return $wpdb->query($query);
	}

	/**
	 * LOGS EXPLORER
	 */
	public function fixPath($path) {
		$path = str_replace('..', '', $path);
		$path = str_replace('//', '/', $path);
		$path = realpath($path);
		$path = rtrim($path, '/').'/';
		return $path;
	}

	public function logPath($value = null) {
		if (is_null($value)) { // GET
			$path = get_option(PLATON_CDN_LOG_PATH, '');
			$path = $this->fixPath($path);
			return $path;
		} else { // SET
			return update_option(PLATON_CDN_LOG_PATH, $value);
		}
	}

	public function logFiles() {
		$logpath = $this->logPath();
		if (is_null(($logpath))
			|| !file_exists($logpath))
		{
			return array();
		}
		$files = scandir($logpath);
		$ret = array();
		if (is_array($files)) foreach ($files as $key => $file) {
			$filepath = $logpath.$file;
			if (!is_file($filepath)) continue;
			$ret[] = array(
				'name' => $file,
				'path' => $filepath,
				'size' => filesize($filepath),
				'mtime' => filemtime($filepath)
			);
		}
		return $ret;
	}

	public function logFetch($filename) {
		$logpath = $this->logPath();
		$filepath = $logpath.$filename;
		if (!file_exists($filepath)
			|| !is_file($filepath))
		{
			return array('status' => 'ERROR', 'message' => 'File not exists.');
		}
		if (substr($filepath, -4) == '.bz2') {
			if (!function_exists('bzopen')) {
				return array('status' => 'ERROR', 'message' => 'bz2 extension is not enabled.');
			}
			$bz = bzopen($filepath, "r");
			if ($bz === false) {
				return array('status' => 'ERROR', 'message' => 'Couldn\'t open file.');
			}
			$decompressed_file = '';
			while (!feof($bz)) {
				$decompressed_file .= bzread($bz, 4096);
			}
			bzclose($bz);
			$content = $decompressed_file;
		} else {
			$content = file_get_contents($filepath);
		}
		if (empty($content)) {
			return array('status' => 'ERROR', 'message' => 'File is empty.');
		}
		$content = strip_tags($content);
		$content = $this->convertBashColorsToHTML($content);
		return array(
			'status' => 'OK',
			'content' => $content
		);
	}

	public function convertBashColorsToHTML($text) {
		$dictionary = array(
			'[m'   => '',
			'[30m' => '<span style="color:black">',
			'[31m' => '<span style="color:red">',         '[91m' => '<span style="color:lightred">',
			'[32m' => '<span style="color:green">',       '[92m' => '<span style="color:lightgreen">',
			'[33m' => '<span style="color:yellow">',      '[93m' => '<span style="color:lightyellow">',
			'[34m' => '<span style="color:blue">',        '[94m' => '<span style="color:lightblue">',
			'[35m' => '<span style="color:purple">',      '[95m' => '<span style="color:lightpurple">',
			'[36m' => '<span style="color:cyan">',        '[96m' => '<span style="color:lightcyan">',
			'[37m' => '<span style="color:lightgray">',   '[97m' => '<span style="color:white">',
			'[90m' => '<span style="color:gray">',
			'[0m'   => '</span>'
		);
		$htmlString = str_replace(array_keys($dictionary), $dictionary, $text);
		return $htmlString;
	}

}

?>
