<?php

if (strtolower(php_sapi_name()) != 'cli') {
	die('This script can only be run from the command line.');
}

ob_start();

/** Wordpress bluff */
if (!isset($_SERVER['REQUEST_METHOD'])) {
	$_SERVER['REQUEST_METHOD'] = 'GET';
}

require_once(explode("wp-content", __FILE__)[0] . "wp-load.php");
require_once __DIR__.'/lib/SFTPconnection.php';
require_once __DIR__.'/lib/PlatonCDN.php';

/** Wordpress corrections */
global $wpdb;
$wpdb->suppress_errors(true);
add_filter('wp_die_handler', 'get_platon_die_handler' );

/** Constants */
date_default_timezone_set('Europe/Bratislava');

if (!defined('PLATON_CDN_DB_RECONNECT_TRY')) {
	define('PLATON_CDN_DB_RECONNECT_TRY', 20);
}
if (!defined('PLATON_CDN_DB_RECONNECT_DELAY')) {
	define('PLATON_CDN_DB_RECONNECT_DELAY', 30);
}

define('COLOR_BLACK', 30);
define('COLOR_RED', 31);              define('COLOR_BRIGHT_RED', 91);
define('COLOR_GREEN', 32);            define('COLOR_BRIGHT_GREEN', 92);
define('COLOR_YELLOW', 33);           define('COLOR_BRIGHT_YELLOW', 93);
define('COLOR_BLUE', 34);             define('COLOR_BRIGHT_BLUE', 94);
define('COLOR_MAGENTA', 35);          define('COLOR_BRIGHT_MAGENTA', 95);
define('COLOR_CYAN', 36);             define('COLOR_BRIGHT_CYAN', 96);
define('COLOR_WHITE', 37);            define('COLOR_BRIGHT_WHITE', 97);
define('COLOR_GRAY', 90);

/** Parse arguments */
$shortopts  = "";
$shortopts .= "h"; // help
$shortopts .= "v"; // version
$shortopts .= "i::"; // disable-steps
$shortopts .= "d::"; // date-like
$shortopts .= "c::"; // count
$shortopts .= "f::"; // fork
$shortopts .= "t::"; // to-time
$shortopts .= "m::"; // max-time
$shortopts .= "p::"; // post-processing
$shortopts .= "u::"; // unselect-no-files
$shortopts .= "s"; // simulation
$shortopts .= "r"; // run-in-cron
$shortopts .= "l"; // coloured

$longopts  = array(
	"help",
	"version",
	"disable-steps::",
	"date-like::",
	"count::",
	"fork::",
	"to-time::",
	"max-time::",
	"post-processing::",
	"unselect-no-files::",
	"simulation::",
	"run-in-cron",
	"coloured",
	"test"
);
$options = getopt($shortopts, $longopts);

if (!is_null(optionParse($options, array('h', 'help')))) {
	printHelp();
}
if (!is_null(optionParse($options, array('v', 'version')))) {
	printVersion();
}
$disable_steps = optionParse($options, array('i', 'disable-steps'));
$date_like = optionParse($options, array('d', 'date-like'));
$count = optionParse($options, array('c', 'count'));
$fork_max_processes = optionParse($options, array('f', 'fork'));
$use_fork = !is_null($fork_max_processes);
$to_time = optionParse($options, array('t', 'to-time'));
$max_time = optionParse($options, array('m', 'max-time'));
$post_processing = optionParse($options, array('p', 'post-processing'));
$unselect_no_files = optionParse($options, array('u', 'unselect-no-files'));
$simulation = !is_null(optionParse($options, array('s', 'simulation')));
$run_in_cron = !is_null(optionParse($options, array('r', 'run-in-cron')));
$coloured = !is_null(optionParse($options, array('l', 'coloured')));
$test = !is_null(optionParse($options, array('test')));

if (is_null($disable_steps)) {
	$disable_steps = '';
}

if ($test) {
	runTests();
	exit(0);
}

$to_time_ts = null;
if (!is_null($to_time)) {
	$to_time_ts = strtotime($to_time);
}
if (!is_null($max_time)) {
	$seconds = abs(intval($max_time));
	$to_time_ts = Time() + $seconds;
}

/** Check requirements */
if (!function_exists('ssh2_connect')) {
	writeline('Error: ssh2_connect function is not available. Try to install php-ssh2 extension. For example: apt-get install php-ssh2', COLOR_RED);
	die();
}

if ($use_fork) {
	if (!function_exists('pcntl_fork')) {
		writeline('Error: pcntl_fork function is not available. Try to install php-pcntl extension. For example: apt-get install php-pcntl', COLOR_RED);
		die();
	}
	writeline('➡️ Fork mode enabled with maximim processes = '.$fork_max_processes, COLOR_GREEN);
}
if ($simulation) {
	writeline('➡️ Simulation mode enabled.', COLOR_GREEN);
}
if ($run_in_cron) {
	writeline('➡️ Run in cron mode enabled.', COLOR_GREEN);
}
if (!is_null($to_time_ts)) {
	writeline('Time limit set to '.date('Y-m-d H:i:s', $to_time_ts), COLOR_GREEN);
}
write('➡️ Current site URL is ');
$site_url = get_site_url();
writeline($site_url, COLOR_YELLOW);

$wp_upload = wp_upload_dir();
$upload_basedir = $wp_upload['basedir'];
$upload_url = $wp_upload['baseurl'];

/** Connection CDN */
$platonCDN = new PlatonCDN();
$cdn_id = $platonCDN->connectionSelected();
if ($cdn_id <= 0) {
	writeline('No CDN connection selected.', COLOR_RED);
	die();
}
$connection_cdn = $platonCDN->connectionGet($cdn_id);
$connection = CDNconnect($connection_cdn);

$prefixes_all = $platonCDN->prefixes();
$prefixes = array();
if (is_array($prefixes_all)) foreach ($prefixes_all as $key => $prefix) {
	$prefixes[] = $prefix['content'];
}

$conversion_webp = $platonCDN->conversionEnabled();
$conversion_quality = $platonCDN->conversionQuality();

/**
 * ----------------------------------------------------------------------------------------------------
 * UNSELECT NO FILES
 */
if (!is_null($unselect_no_files)) {
	write('Unselect #no-files# with date search = '.$unselect_no_files, COLOR_GREEN);
	writeTimeBegin();
	$stats = array('processed' => 0, 'removed_no_files_ok' => 0, 'removed_no_files_failed' => 0);

	$run_loop = true;
	while ($run_loop) {
		if (!is_null($count) && $count <= 0) {
			writeline('All images processed.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		if (!is_null($count)) {
			$count--;
		}
		if (!is_null($to_time_ts) && Time() > $to_time_ts) {
			writeline('Time limit reached.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		writeline();
		write('Search next image ... ');
		$image = getImageNoFiles($unselect_no_files);
		if (!is_array($image)
			|| count($image) <= 0)
		{
			writeline('No images for processing.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		$stats['processed']++;
		// print_r($image);
		write('Post ID: ');
		writeline($image['ID'], COLOR_BRIGHT_BLUE);
		write('- file URL: ');
		writeline($image['guid'], COLOR_YELLOW);
		write('- remove flag #no-files# ... ');
		$guid_new = str_replace('#no-files#', '', $image['guid']);
		if ($simulation || updateGUID($image['ID'], $guid_new)) {
			writeline('OK', COLOR_BRIGHT_GREEN);
			$stats['removed_no_files_ok']++;
		} else {
			writeline('FAILED', COLOR_BRIGHT_RED);
			$stats['removed_no_files_failed']++;
		}

		$check_metadata = false;
		$check_filesystem = false;
		$check_using = false;

		write('- check exists metadata in table wp_postmeta ... ');
		switch (checkMetadataExists($image['ID'])) {
			default: writeline('FAILED, unknown status', COLOR_RED); break;
			case 1: writeline('OK', COLOR_GREEN); break;
			case 0: writeline('FAILED, metadata is missing', COLOR_RED); break;
			case -1: writeline('FAILED, metadata is corrupted', COLOR_RED); break;
		}

		write('- get metadata by wordpress ... ');
		$metadata = wp_get_attachment_metadata($image['ID']);
		$files = array();
		$files_exists = array();
		if (is_array($metadata)) {
			writeline('OK', COLOR_GREEN);
			printWordpressAttachmentMetadata($metadata);
			$files = extractFilesFromMeta($metadata);
			if (count($files) > 0) {
				$check_metadata = true;
			}
			foreach ($files as $file) {
				write('  - check file exists ');
				write($file, COLOR_BRIGHT_CYAN);
				write(' ... ');
				if (!file_exists($upload_basedir.'/'.$file)) {
					writeline('FAIL, file not found.', COLOR_RED);
				} else {
					writeline('OK', COLOR_GREEN);
					$files_exists[] = $upload_basedir.'/'.$file;
				}
			}
		} else {
			writeline('FAILED, metadata is not array', COLOR_RED);
		}

		write('- check main image from GUID on filesystem ... ');
		$imagepath = extractImagePathFromURL($image['guid']);
		if (!is_null($imagepath)) {
			writeline('extract image path from GUID successfully', COLOR_GREEN);
			write('  - check filepath: ');
			$filepath = concatPath($upload_basedir, $imagepath);
			write($filepath, COLOR_YELLOW);
			write(' ... ');
			if (file_exists($filepath)) {
				writeline('OK', COLOR_GREEN);
				$check_filesystem = true;
				if (!in_array($filepath, $files_exists)) {
					$files_exists[] = $filepath;
				}
			} else {
				writeline('FAILED, file not found on filesystem', COLOR_RED);
			}
		} else {
			writeline('FAILED, not found image path', COLOR_RED);
		}

		if (count($files) <= 0
			&& !is_null($imagepath))
		{
			$files[] = $imagepath;
		}
		if (count($files) <= 0) {
			writeline('- no images for search in articles', COLOR_BRIGHT_MAGENTA);
		} else {
			writeline('- check images using in articles');
		}
		if ($use_fork) {
			$posts_with_images = getPostWithImages3Fork($files);
		} else {
			$posts_with_images = getPostWithImages3($files);
		}
		if (!is_array($posts_with_images)
			|| count($posts_with_images) <= 0)
		{
			writeline('No posts with image found.', COLOR_RED);
		} else {
			$posts_count = 0;
			foreach ($posts_with_images as $file => $posts) {
				write('  - for image ');
				write($file, COLOR_YELLOW);
				writeline(' founded in posts: ');
				if (count($posts) > 0) {
					$check_using = true;
					foreach ($posts as $post) {
						write('    [ID: ');
						write($post['ID'], COLOR_BRIGHT_BLUE);
						write('] ');
						$post_url = get_page_link($post['ID']);
						writeline($post_url, COLOR_BRIGHT_YELLOW);
					}
				} else {
					writeline('    Not found in articles.', COLOR_BRIGHT_MAGENTA);
				}
			}
		}

		write('- summary of checks: metadata = ');
		writeCheck($check_metadata);
		write(', file system = ');
		writeCheck($check_filesystem);
		write(', using in articles = ');
		writeCheck($check_using);
		writeline();
		write('- recommendation: ');
		$check_code = ($check_metadata ? '1' : '0') . ($check_filesystem ? '1' : '0') . ($check_using ? '1' : '0');
		switch($check_code) {
			case '000':
				writeline('Remove post ID with image from DB.', COLOR_BRIGHT_WHITE);
				writeline('  '.$site_url.'/wp-admin/upload.php?item='.$image['ID'], COLOR_GRAY);
				//writeline('  DELETE FROM wp_posts WHERE ID = '.$image['ID'].';', COLOR_GRAY);
				//writeline('  DELETE FROM wp_postmeta WHERE post_id = '.$image['ID'].';', COLOR_GRAY);
				break;
			case '001':
				writeline('Create a image and upload on CDN, then run post-processing.', COLOR_BRIGHT_WHITE);
				writeline('  No command available, only manual fix.', COLOR_GRAY);
				break;
			case '010':
				writeline('Remove image from file system and clear metadata.', COLOR_BRIGHT_WHITE);
				if (count($files_exists) > 0) {
					writeline('  rm '.implode(' ', $files_exists), COLOR_GRAY);
				}
				writeline('  '.$site_url.'/wp-admin/upload.php?item='.$image['ID'], COLOR_GRAY);
				break;
			case '011':
				//writeline('Upload image from file system to CDN, then run post-processing.', COLOR_BRIGHT_WHITE);
				//writeline('  No command available, only manual fix.', COLOR_GRAY);
				writeline('Image is on CDN and article has valid link after post-processing, then you can remove image from Admin.', COLOR_BRIGHT_WHITE);
				writeline('  '.$site_url.'/wp-admin/upload.php?item='.$image['ID'], COLOR_GRAY);
				break;
			case '100':
				writeline('Remove metadata from DB.', COLOR_BRIGHT_WHITE);
				writeline('  '.$site_url.'/wp-admin/upload.php?item='.$image['ID'], COLOR_GRAY);
				//writeline('  DELETE FROM wp_posts WHERE ID = '.$image['ID'].';', COLOR_GRAY);
				//writeline('  DELETE FROM wp_postmeta WHERE post_id = '.$image['ID'].';', COLOR_GRAY);
				break;
			case '101':
				writeline('Create a image, upload on filesystem and CDN, then run post-processing.', COLOR_BRIGHT_WHITE);
				writeline('  No command available, only manual fix.', COLOR_GRAY);
				break;
			case '110':
				writeline('Remove image from gallery in wordpress admin.', COLOR_BRIGHT_WHITE);
				writeline('  '.$site_url.'/wp-admin/upload.php?item='.$image['ID'], COLOR_GRAY);
				break;
			case '111':
				writeline('All right', COLOR_GREEN);
				break;
		}

	}

	writeTimeEnd();
	printStats($stats);
 	exit(0);
}

/**
 * ----------------------------------------------------------------------------------------------------
 * POST PROCESSING
 */
if (!is_null($post_processing)) {
	write('Post processing mode enabled with search after upload URL = '.$post_processing, COLOR_GREEN);
	writeTimeBegin();
	$stats = array('processed_posts' => 0,
		'founded_links_all' => 0, 'founded_links_matched' => 0, 'replaced_links' => 0,
		'updated_posts_ok' => 0, 'updated_posts_failed' => 0, 'posts_no_changes' => 0,
		'check_url_ok' => 0, 'check_url_failed' => 0);
	$run_loop = true;
	$post_id_min = getPostIdMin();
	$post_id_max = getPostIdMax();
	$post_id_current = null;
	$upload_url_prefixes = array();
	foreach ($prefixes as $prefix) {
		$upload_url_prefixes[] = concatURL($prefix, $post_processing);
	}
	writeline('Search for URL contains ');
	foreach ($upload_url_prefixes as $index => $url_prefix) {
		writeline('['.$index.'] '.$url_prefix, COLOR_YELLOW);
	}
	getNextPostId(true); // reset
	$check_url_faileds = array();
	$check_url_failes_in_posts = array();
	while ($run_loop) {
		if (!is_null($count) && $count <= 0) {
			writeline('All images processed.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		if (!is_null($count)) {
			$count--;
		}
		if (!is_null($to_time_ts) && Time() > $to_time_ts) {
			writeline('Time limit reached.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		//$post_id_current = is_null($post_id_current) ? $post_id_min : $post_id_current + 1;
		$post_id_current = getNextPostId();
		if ($post_id_current > $post_id_max) {
			writeline('Max. post ID is reached.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		if ($post_id_current === false) {
			writeline('No more posts found.', COLOR_GREEN);
			$run_loop = false;
			break;
		}
		writeline();
		write('Processing post with ID = ');
		write($post_id_current.' ', COLOR_BRIGHT_BLUE);
		$post = getPostById($post_id_current);
		if (empty($post)) {
			writeline('- No post found', COLOR_BRIGHT_MAGENTA);
			continue;
		}
		$stats['processed_posts']++;
		$content = $post['post_content'];
		$regex = "/\b(?:(?:https?):\/\/|www\.)[-–a-z0-9+&@#\/%?=~_|!:,.;´\p{L}\p{M}]*[-–a-z0-9+&@#\/%=~_|´\p{L}\p{M}]/iu";
		preg_match_all($regex, $content, $matches);
		//print_r($matches);
		if (empty($matches[0])) {
			writeline('- No links found', COLOR_GREEN);
			continue;
		}
		$founded_links = $matches[0];
		write('- Founded links count = ');
		writeline(count($founded_links), COLOR_BRIGHT_BLUE);
		$stats['founded_links_all'] += count($founded_links);
		foreach ($founded_links as $link) {
			write('- For link: ');
			write($link.' ', COLOR_YELLOW);
			$upload_url_simple = null;
			foreach ($upload_url_prefixes as $url_prefix) {
				if (stristr($link, $url_prefix) !== false) {
					$upload_url_simple = $url_prefix;
				}
			}
			if (is_null($upload_url_simple)) {
				writeline('skipped', COLOR_MAGENTA);
				continue;
			}
			$stats['founded_links_matched']++;
			writeline();
			$local_parts = explode($upload_url_simple, $link);
			$remote_url = concatURL($connection_cdn['upload_url'], $post_processing, $local_parts[1]);
			if ($conversion_webp) {
				$remote_url .= '.webp';
			}
			write('  - Replacing to: ');
			writeline($remote_url, COLOR_BRIGHT_YELLOW);
			write('  - Checking remote URL ... ');
			if ($simulation || checkURL2($remote_url, $error_msg)) {
				writeline('OK', COLOR_GREEN);
				$content = str_replace($link, $remote_url, $content);
				$stats['check_url_ok']++;
				$stats['replaced_links']++;
			} else {
				writeline('FAILED, URL will not be replaced', COLOR_RED);
				write('  - check URL failed with error message: ');
				writeline($error_msg, COLOR_BRIGHT_RED);
				$stats['check_url_failed']++;
				if (!in_array($remote_url, $check_url_faileds)) {
					$check_url_faileds[] = $remote_url;
				}
				if (!is_array($check_url_failes_in_posts[$remote_url])) {
					$check_url_failes_in_posts[$remote_url] = array(
						'post_ids' => array(),
						'url_postfix' => concatURL($post_processing, $local_parts[1])
					);
				}
				if (!in_array($post_id_current, $check_url_failes_in_posts[$remote_url]['post_ids'])) {
					$check_url_failes_in_posts[$remote_url]['post_ids'][] = $post_id_current;
				}
			}
		}
		if ($content != $post['post_content']) {
			write('- Updating post content ... ');
			$suc = $simulation || updatePostContent($post_id_current, $content);
			if ($suc) {
				writeline('OK', COLOR_GREEN);
				$stats['updated_posts_ok']++;
			} else {
				writeline('FAIL', COLOR_RED);
				$stats['updated_posts_failed']++;
			}
		} else {
			writeline('- No changes in post content', COLOR_BRIGHT_MAGENTA);
			$stats['posts_no_changes']++;
		}
	}
	write('Post-proceesing is done. ', COLOR_GREEN);
	writeTimeEnd();
	if (!empty($check_url_faileds)) {
		writeline('Summary of failed check URLs (without duplicate):', COLOR_RED);
		foreach ($check_url_faileds as $link) {
			writeline('- '.$link, COLOR_BRIGHT_RED);
			if (is_array($check_url_failes_in_posts[$link])) {
				writeline('  - in post IDs: '.implode(', ', $check_url_failes_in_posts[$link]['post_ids']));
				foreach ($check_url_failes_in_posts[$link]['post_ids'] as $failed_url_post_id) {
					write('    [ID: '.$failed_url_post_id.'] ');
					writeline(get_page_link($failed_url_post_id), COLOR_YELLOW);
				}
				write('  - search image post ID ... ');
				$image_post = getLocalImages(1, array('GUID' => '%'.$check_url_failes_in_posts[$link]['url_postfix'].'%'), false);
				if (is_array(($image_post))
					&& count($image_post) > 0)
				{
					writeline($image_post[0]['ID'], COLOR_BRIGHT_BLUE);
					$image_meta = wp_get_attachment_metadata($image_post[0]['ID']);
					printWordpressAttachmentMetadata($image_meta);
				} else {
					writeline('NOT FOUND', COLOR_BRIGHT_MAGENTA);
				}
			}

		}
	}
	printStats($stats);
	exit(0);
}

/**
 * ----------------------------------------------------------------------------------------------------
 * MAIN PROCESSING
 */

$stats = array(
	'processed_images' => 0,
	'no_files_to_upload' => 0,
	'file_not_found' => 0,
	'uploaded_to_cdn' => 0,
	'files_not_found_on_cdn' => 0,
	'update_attachment_guid_failed' => 0,
	'update_attachment_file_failed' => 0,
	'delete_local_file_failed' => 0,
);
$run_loop = true;
while ($run_loop) {
	// 0. kontrola poctu spracovanych obrazkov
	if (!is_null($count) && $count <= 0) {
		writeline('All images processed.', COLOR_GREEN);
		$run_loop = false;
		cache_flush();
		break;
	}
	if (!is_null($count)) {
		$count--;
	}
	if (!is_null($to_time_ts) && Time() > $to_time_ts) {
		writeline('Time limit reached.', COLOR_GREEN);
		$run_loop = false;
		cache_flush();
		break;
	}
	// 1. vyhladat v tabulke obrazok, ktory je lokalny
	if (stristr($disable_steps, '1') !== false) {
		writeline('Step 1 disabled, processing can not continue', COLOR_GREEN);
		$run_loop = false;
		break;
	}
	$where = array();
	if (!is_null($date_like)) {
		$where['post_date'] = $date_like;
	}
	writeline();
	writeTimeBegin();
	write('Searching for next image ... ');
	$images = getLocalImages(1, $where);
	if (empty($images)) {
		writeline('No images to process.', COLOR_GREEN);
		$run_loop = false;
		cache_flush();
		break;
	} else {
		writeline('OK', COLOR_GREEN);
	}
	$image = $images[0]; // print_r($image);
	$image_modified_gmt = $image['post_modified_gmt'];
	writeTimestamp();
	write('Processing image with post ID ');
	write('#'.$image['ID'], COLOR_BRIGHT_BLUE);
	write('- GUID: ');
	writeline($image['guid'], COLOR_CYAN);
	//$meta = getPostMeta($image['ID']);
	$meta = wp_get_attachment_metadata($image['ID']);
	$files = extractFilesFromMeta($meta);
	if (empty($files)) {
		write('- FAIL: No files to upload. Append flag #no-files# to GUID. ', COLOR_RED);
		$stats['no_files_to_upload']++;
		if ($simulation || updateGUID($image['ID'], $image['guid'].'#no-files#')) {
			writeline('GUID updated to #no-files#', COLOR_GREEN);
		} else {
			writeline('GUID update to #no-files# failed.', COLOR_RED);
		}
		continue;
	} else {
		foreach ($files as $file) {
			write('- check file exists ');
			write($file, COLOR_BRIGHT_CYAN);
			write(' ... ');
			if (!file_exists($upload_basedir.'/'.$file)) {
				write('FAIL, file not found. Append flag #no-files# to GUID ... ', COLOR_RED);
				$stats['file_not_found']++;
				if ($simulation || updateGUID($image['ID'], $image['guid'].'#no-files#')) {
					writeline('OK', COLOR_GREEN);
				} else {
					writeline('FALIED', COLOR_RED);
				}
				$files = array(); // reset files
				break;
			} else {
				writeline('OK', COLOR_GREEN);
			}
		}
	}
	if (!is_array($files)
		|| empty($files)
		|| count($files) <= 0)
	{
		writeline('- FAIL: No files of image for processing.');
		continue;
	}

	// 2. (nejako / cez SFTP) uploadnut na CDN, spolu s obrazkom aj jeho resiznute velkosti (su spomenute v wp_postmeta)
	if (stristr($disable_steps, '2') === false) foreach ($files as $file) {
		if ($use_fork) {
			$connection = CDNconnect($connection_cdn); // po fork procesu sa pokazi session na SFTP, preto sa pripajame znovu
		}
		write('- Uploading file: ');
		write($file, COLOR_BRIGHT_CYAN);
		if (!file_exists($upload_basedir.'/'.$file)) {
			writeline('... FAIL: File not found, skipped.', COLOR_RED);
			continue;
		}
		$local_file = $upload_basedir.'/'.$file;
		$remote_file = $connection_cdn['upload_dir'].'/'.$file;
		$remove_file_webp = $remote_file.'.webp';
		$size = sizeUserFriendly(filesize($local_file));
		write(' [');
		write($size, COLOR_BRIGHT_WHITE);
		write('] ... ');
		/* TODO otestovat
		if ($simulation || $connection->fileExists($remote_file)) {
			writeline('file is already on CDN', COLOR_GREEN);
			continue;
		}
		*/
		// Upload standardneho suboru
		$suc = false; $max_try_again = 3;
		while (!$suc && $max_try_again-- > 0) {
			try {
				$suc = $simulation || $connection->uploadFile($local_file, $remote_file);
				if (!$suc) {
					writeline('Upload to CDN failed.', COLOR_RED);
					die();
				} else {
					writeline('OK', COLOR_GREEN);
					$stats['uploaded_to_cdn']++;
				}
			} catch (Exception $e) {
				writeline($e->getMessage(), COLOR_RED);
				if ($max_try_again > 0) {
					write('  Wait 1 seconds and try reconnect again ... ', COLOR_BRIGHT_RED);
					sleep(1);
					$connection = CDNconnect($connection_cdn);
				} else {
					write('  Max try again reached. ', COLOR_BRIGHT_RED);
				}
			}
		}
		if (!$suc) {
			writeline('Upload to CDN failed.', COLOR_RED);
			die();
		}
		// Upload WEBP suboru
		if ($conversion_webp
			&& substr($file, -5) != '.webp')
		{
			$webp_content = $platonCDN->webpImage($local_file, $conversion_quality);
			if ($webp_content !== false) {
				write('- Uploading file: ');
				write($file.'.webp', COLOR_BRIGHT_CYAN);
				$size_webp = sizeUserFriendly(strlen($webp_content));
				write(' [');
				write($size_webp, COLOR_BRIGHT_WHITE);
				write('] ... ');
				$suc = false; $max_try_again = 3;
				while (!$suc && $max_try_again-- > 0) {
					try {
						$suc = $simulation || $connection->uploadFileContent($webp_content, $remove_file_webp);
						if (!$suc) {
							writeline('Upload to CDN failed.', COLOR_RED);
							die();
						} else {
							writeline('OK', COLOR_GREEN);
							$stats['uploaded_to_cdn']++;
						}
					} catch (Exception $e) {
						writeline($e->getMessage(), COLOR_RED);
						if ($max_try_again > 0) {
							write('  Wait 1 seconds and try reconnect again ... ', COLOR_BRIGHT_RED);
							sleep(1);
							$connection = CDNconnect($connection_cdn);
						} else {
							write('  Max try again reached. ', COLOR_BRIGHT_RED);
						}
					}
				}
				if (!$suc) {
					writeline('Upload to CDN failed.', COLOR_RED);
					die();
				}
			} else {
				writeline('Convert to WEBP failed.', COLOR_RED);
			}
		}

	}

	//echo "sleep \n"; sleep(10);

	// 3. overit cez CURL ci je obrazok dostupny na CDN
	$failed = 0;
	$file_to_cdn_url = array();
	$url_to_replace = array();
	foreach ($files as $file) {
		$check_urls = array();
		$remote_url = $connection_cdn['upload_url'].'/'.$file;
		$check_urls[] = $remote_url;
		if ($conversion_webp) {
			$remote_url = $connection_cdn['upload_url'].'/'.$file.'.webp';
			$check_urls[] = $remote_url;
		}
		$file_to_cdn_url[$file] = $remote_url;
		$file_to_cdn_url[basename($file)] = $remote_url;
		foreach ($prefixes as $prefix) {
			$url_to_replace[concatURL($prefix, $file)] = $remote_url;
		}
		if (stristr($disable_steps, '3') === false) {
			foreach ($check_urls as $check_url) {
				write('- Checking file on URL: ');
				write($check_url, COLOR_YELLOW);
				write(' ... ');
				$is_image_on_cdn = $simulation || checkURL2($check_url, $error_msg);
				if ($is_image_on_cdn) {
					writeline('OK', COLOR_GREEN);
				} else {
					writeline('FAIL', COLOR_RED);
					write('- check URL failed with error message: ');
					writeline($error_msg, COLOR_BRIGHT_RED);
					$failed++;
				}
			}
		}
	}
	if ($failed > 0) {
		writeline("- FAIL: Some files not found on CDN. Append flag #no-files# to GUID ...", COLOR_RED);
		$stats['files_not_found_on_cdn']++;
		if ($simulation || updateGUID($image['ID'], $image['guid'].'#no-files#')) {
			writeline('OK', COLOR_GREEN);
		} else {
			writeline('FALIED', COLOR_RED);
		}
		continue;
	}

	// 5. vyhladat clanky, v ktorych je pouzity obrazok s ID a updatnut lokaciu z CDN
	if (stristr($disable_steps, '5') === false) {
		writeTimestamp();
		write('- Searching posts with images ... ');
		// $use_from_date = $run_in_cron ? $image_modified_gmt : null;
		$use_from_date = $image_modified_gmt;
		if ($use_fork) {
			$posts_with_images = getPostWithImages3Fork(array_keys($url_to_replace), $use_from_date);
		} else {
			$posts_with_images = getPostWithImages4(array_keys($url_to_replace), $use_from_date);
		}
		if (!is_array($posts_with_images)) {
			writeline('No posts with image found.', COLOR_RED);
		} else {
			$posts_count = 0;
			foreach ($posts_with_images as $file => $posts) {
				$posts_count += count($posts);
			}
			writeline('count = '.$posts_count, COLOR_BRIGHT_BLUE);
		}
		writeTimestamp();
	} else {
		$posts_with_images = array(); // no for processing
	}
	$failed = 0;
	$post_ids_updated = array();
	foreach ($posts_with_images as $file => $posts) {
		write('- For image ');
		write($file, COLOR_BRIGHT_CYAN);
		write(' found ');
		writeline(count($posts).' posts', COLOR_BRIGHT_BLUE);
		foreach ($posts as $key => $post) {
			write('  - Updating post ID ');
			write('#'.$post['ID'], COLOR_BRIGHT_BLUE);
			write(' ... ');
			if (in_array($post['ID'], $post_ids_updated)) {
				writeline('UPDATED ABOVE', COLOR_GREEN);
				continue;
			}
			$content = $post['post_content'];
			$new_content = str_replace(array_keys($url_to_replace), array_values($url_to_replace), $content);
			if ($content != $new_content) {
				$suc = $simulation || updatePostContent($post['ID'], $new_content);
				if ($suc) {
					writeline('OK', COLOR_GREEN);
					$post_ids_updated[] = $post['ID'];
				} else {
					writeline('FAIL', COLOR_RED);
					$failed++;
				}
			} else {
				writeline('NO CHANGES', COLOR_RED);
			}
			$post_url = get_permalink($post['ID']);
			write('  - Post URL: ');
			writeline($post_url, COLOR_BRIGHT_YELLOW);
		}
	}
	if ($failed > 0) {
		writeline("Some posts failed to update.", COLOR_RED);
		die();
		break;
	}

	// 4. updatnut metadata pre obrazok
	if (stristr($disable_steps, '4') === false) {
		writeTimestamp();
		// 4.1 updatnut wp_post guid na CDN URL
		write('- Updating attachment GUID ... ');
		$remote_url = $connection_cdn['upload_url'].'/'.$meta['file'];
		if ($conversion_webp) {
			$remote_url = $connection_cdn['upload_url'].'/'.$meta['file'].'.webp';
		}
		if ($simulation || updateGUID($image['ID'], $remote_url)) {
			writeline('OK', COLOR_GREEN);
		} else {
			writeline('FAIL', COLOR_RED);
			$stats['update_attachment_guid_failed']++;
		}

		// 4.2 updatenut wp_postmeta pre konkretny obrazok
		write('- Updating attachment file ... ');
		if ($simulation || updateAttachmentFile($image['ID'], $file_to_cdn_url)) {
			writeline('OK', COLOR_GREEN);
		} else {
			writeline('FAIL', COLOR_RED);
			$stats['update_attachment_file_failed']++;
		}
		// echo "- Updating attachment metadata ... ";
		// if (updateAttachmentMetadata($image['ID'], $file_to_cdn_url)) {
		// 	echo "OK\n";
		// } else {
		// 	echo "FAIL\n";
		// }
	}

	// 6. moze sa lokalny obrazok a jeho resiznute subory zmazat
	if (stristr($disable_steps, '6') === false) foreach ($files as $file) {
		$local_file = $upload_basedir.'/'.$file;
		write('- Deleting local file: ');
		write($local_file, COLOR_BRIGHT_MAGENTA);
		write(' ... ');
		if (!file_exists($local_file)) {
			writeline('OK', COLOR_GREEN);
			continue;
		}
		if ($simulation || unlink($local_file)) {
			writeline('OK', COLOR_GREEN);
		} else {
			writeline('FAIL', COLOR_RED);
			$stats['delete_local_file_failed']++;
		}
	}

	write('Image processed ');
	writeTimeEnd();
	$stats['processed_images']++;

	//$run_loop = false; // for testing with one image
	echo "\n"; //blank line for loop separation
}
printStats($stats);

/**
 * ----------------------------------------------------------------------------------------------------
 * FUNCTIONS
 */
$_write = '';
$_stderr_enabled = true;

function stderr_enable() {
	global $_stderr_enabled;
	$_stderr_enabled = true;
}

function stderr_disabled() {
	global $_stderr_enabled;
	$_stderr_enabled = false;
}

function write($msg, $color_code = null) {
	global $coloured, $run_in_cron, $_stderr_enabled;
	if ($coloured) {
		echo "\e[".$color_code."m";
	}
	echo $msg;
	if ($coloured) {
		echo "\e[0m";
	}
	if ($_stderr_enabled
		&& $run_in_cron
		&& stristr($msg, 'fail') !== false)
	{
		fwrite(STDERR, trim($msg)."\n");
	}
	flush();
	ob_flush();
}

function writeline($msg = '', $color_code = null) {
	return write($msg."\n", $color_code);
}

function writeCheck($bool) {
	if ($bool) {
		write('OK', COLOR_GREEN);
	} else {
		write('FAILED', COLOR_RED);
	}
}

$_time_begin = null;
function writeTimeBegin() {
	global $_time_begin;
	$_time_begin = Time();
	write('⌚ [Time Begin = ');
	write(date('Y-m-d H:i:s', $_time_begin), COLOR_BRIGHT_WHITE);
	writeline('] ');
}

function writeTimeEnd() {
	global $_time_begin;
	$time_end = Time();
	write('⌚ [Time End = ');
	write(date('Y-m-d H:i:s', $time_end), COLOR_BRIGHT_WHITE);
	write('] in ');
	$time_total = $time_end - $_time_begin;
	$time_minutes = floor($time_total / 60);
	$time_seconds = $time_total % 60;
	writeline($time_minutes.' min '.$time_seconds.' s', COLOR_BRIGHT_WHITE);
}

$_time_last = null;
function writeTimestamp() {
	global $_time_begin, $_time_last;
	$now = Time();
	$time_total = $now - ($_time_last ?? $_time_begin);
	$_time_last = $now;
	$time_minutes = floor($time_total / 60);
	$time_seconds = $time_total % 60;
	write('⌚ [Timestamp = ');
	write(date('Y-m-d H:i:s', $now), COLOR_BRIGHT_WHITE);
	write(', Δt = ');
	write($time_minutes.' min '.$time_seconds.' s', COLOR_BRIGHT_WHITE);
	writeline(']');
}

function optionParse($options, $keys, $default = null) {
	foreach ($keys as $key) {
		if (isset($options[$key])) {
			return $options[$key];
		}
	}
	return $default;
}

function printStats($stats) {
	stderr_disabled();
	writeline('Statistics:', COLOR_BLUE);
	$max_length_key = 0;
	foreach ($stats as $key => $value) {
		$max_length_key = max($max_length_key, strlen($key));
	}
	$max_length_key += 2;
	foreach ($stats as $key => $value) {
		$label = str_replace('_', ' ', $key);
		$label = ucwords($label);
		write(sprintf('%'.$max_length_key.'s: ', $label), COLOR_BRIGHT_WHITE);
		$color_text = COLOR_BRIGHT_BLUE;
		if ($value > 0
			&& stristr($key, 'fail') !== false)
		{
			$color_text = COLOR_BRIGHT_RED;
		}
		writeline(sprintf('%8s ', $value), $color_text);
	}
	stderr_enable();
}

function concatURL() {
	$ret = '';
	$parts = func_get_args();
	foreach ($parts as $index => $part) {
		$part = ltrim($part, '/');
		if (strlen($ret) > 0
			&& substr($ret, -1) != '/')
		{
			$ret .= '/';
		}
		$ret .= $part;
	}
	return $ret;
}

function concatPath() {
	$ret = '';
	$parts = func_get_args();
	foreach ($parts as $index => $part) {
		if (strlen($ret) <= 0) {
			$ret = $part;
		}  else {
			$part = ltrim($part, '/');
			if (substr($ret, -1) != '/')
			{
				$ret .= '/';
			}
			$ret .= $part;
		}
	}
	return $ret;
}

function fileMimeType($filepath) {
	$extension = pathinfo($filepath, PATHINFO_EXTENSION);
	$parts = explode('#', $extension);
	return 'image/'.$parts[0];
}

function array_remove(&$arr, $value) {
	if (!is_array($arr)) return $arr;
	foreach ($arr as $key => $val) {
		if ($val == $value) {
			unset($arr[$key]);
		}
	}
	return $arr;
}

function clearDirectory($dir_path) {
	if (!is_dir($dir_path)) {
		return false;
	}
	$files = glob($dir_path.'/*');
	foreach ($files as $file) {
		if (is_file($file)) {
			unlink($file);
		}
	}
	return true;
}

function printHelp() {
	writeline('Backend for Platon CDN Wordpress plugin version '.Platon_CDN_plugin::$version);
	writeline('Usage: php backend.php [options]');
	writeline('Options:');
	writeline('  -h, --help                     Show this help');
	writeline('  -v, --version                  Show version of this plugin aj MD5 HASH of this backend script');
	writeline('  -i, --disable-steps=<NUMBERS>  Disable steps, e.g. 123 to disable 1st, 2nd and 3rd step');
	writeline('                                 Steps: 1 - find local image, 2 - upload to CDN, 3 - check on CDN, 5 - update posts, 4 - update metadata, 6 - delete local files');
	writeline('  -d, --date-like=<DATE>         Date like string');
	writeline('  -c, --count=<NUMBER>           Count of images to process');
	writeline('  -f, --fork=<NUMBER>            Use fork for parallel processing, number is max count of processes');
	writeline('  -t, --to-time=<DATETIME>       Date and time for max time of processing');
	writeline('  -m, --max-time=<SECONDS>       Maximum seconds for processing');
	writeline('  -p, --post-processing=<PREFIX> Prefix after upload URL for post-processing in posts, like step 5');
	writeline('  -u, --unselect-no-files=<DATE> Remove #no-files from images meta-data, parameter is like date');
	writeline('  -s, --simulation               Simulation mode');
	writeline('  -r, --run-in-cron              Special mode for run in CRON');
	writeline('  -l, --coloured                 Coloured output');
	writeline('  --test                         Run tests for debug');
	writeline('');
	writeline('Example:');
	writeline('  # process 5 images from year 2023');
	writeline('  php backend.php -d=2023-% -c=5');
	writeline('');
	writeline('  # process without step 2, 5 and 6 for month June 2024 with coloured output');
	writeline('  php backend.php -i=256 -d=2024-06% -l');
	writeline('');
	writeline('  # post-processing for replace URL in posts for month June 2024 with coloured output');
	writeline('  php backend.php -p=2024/06 -l');
	writeline('');
	writeline('  # unselect images with flags #no-files in step 3 in standard process for month June 2024 with coloured output');
	writeline('  php backend.php -u=2024-06% -l');
	writeline('');

	exit;
}

function printVersion() {
	writeline('Backend for Platon CDN Wordpress plugin version '.Platon_CDN_plugin::$version);
	writeline('MD5 hash of this file backend.php is '.hash_file('md5', __FILE__));
	writeline('');
	exit;
}

function CDNconnect($connection_cdn) {
	$connection = null;
	switch ($connection_cdn['connection_type']) {
		case 'SFTP':
			try {
				$connection = new SFTPconnection($connection_cdn['hostname'], $connection_cdn['port']);
				$suc = $connection->login($connection_cdn['username'], $connection_cdn['password']);
				if (!$suc) {
					writeline('Login to SFTP failed.', COLOR_RED);
					die();
				}
			} catch (Exception $e) {
				writeline('SFTPconnection: '.$e->getMessage(), COLOR_RED);
				die();
			}
			break;
		default:
			writeline('Not implemented connection type.', COLOR_RED);
			die();
	}
	return $connection;
}

// from Control Panel -> MISC
function sizeUserFriendly($size, $precision = 0, $max_ext = null) /* {{{ */
{
	$exts = array('B', 'kB', 'MB', 'GB', 'TB', 'PB');
	$ret = $size;
	$series = 0;
	while ($ret > 1000 && $series < count($exts)) {
		$ret = round($ret / 1024, $precision);
		$series++;
		if (! empty($max_ext) && !strcasecmp($max_ext, $exts[$series])) {
			break;
		}
	}
	$ret_print = (!empty($max_ext) && array_search($max_ext, $exts) > array_search($exts[$series], $exts))
		? round($ret, 0)
		: $ret;
	return $ret_print.' '.$exts[$series];
} /* }}} */

function fixURL($url) {
	$url_info = parse_url($url);
	$url_prefix = $url_info['scheme'].'://'.$url_info['host'].'/';
	$fixed_url = $url_prefix.urlencode(str_replace($url_prefix, '', $url));
	return $fixed_url;
}

function extractImagePathFromURL($url) {
	$parts = explode('wp-content/uploads', $url);
	if (!is_array($parts)) {
		return null;
	}
	if (count($parts) < 2) {
		return false;
	}
	$parts2 = explode('#', $parts[1]);
	return $parts2[0];
}

function checkURL($url) {
	$url = fixURL($url);
	$ch = curl_init($url);
	curl_setopt($ch, CURLOPT_NOBODY, true);
	curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
	curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
	curl_exec($ch);
	$retcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
	curl_close($ch);
	return $retcode == 200;
}

function checkURL2($url, &$error_msg = null) {
	$url = fixURL($url);
	$headers = @get_headers($url);
	if (!is_array($headers)) {
		$error_msg = 'Connection failure';
		return false;
	}
	if (strpos($headers[0], '200') !== false) {
		return true;
	} else {
		return 'HTTP header response: '.$headers[0];
	}
}

function getPostById($post_id) {
	global $wpdb;
	if (empty($post_id)) {
		return null;
	}
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE ID = %d',
		$wpdb->prefix.'posts',
		intval($post_id));
	$post = wpdb_get_row($query);
	return $post;
}

$_next_post_ids = array();
$_next_post_id_index = 0;
function getNextPostId($reset = false) {
	global $wpdb, $_next_post_ids, $_next_post_id_index;
	$post_index_step = 1000;
	if ($reset) {
		$_next_post_ids = array();
		return true;
	}
	while(count($_next_post_ids) <= 0) {
		if ($_next_post_id_index > getPostIdMax()) {
			return false;
		}
		$_next_post_id_index += $post_index_step;
		$query = sprintf('SELECT ID'
			.' FROM %s'
			.' WHERE ID > %d'
			.' AND ID <= %d'
			.' ORDER BY ID ASC',
			$wpdb->prefix.'posts',
			intval($_next_post_id_index - $post_index_step),
			intval($_next_post_id_index));
		$_next_post_ids = wpdb_get_col($query);
	}
	$post_id = array_shift($_next_post_ids);
	return $post_id;
}

function getImageNoFiles($post_date = null, $use_prefixes = false) {
	global $wpdb, $prefixes;
	$conditions = array(
		'post_type = "attachment"',
		'post_mime_type LIKE "image/%"',
		'guid LIKE "%#no-files%"'
	);
	if (!is_null($post_date)
		&& strlen($post_date) > 0)
	{
		if (substr($post_date, 0, 1) == '%'
			|| substr($post_date, -1) == '%')
		{
			$conditions[] = 'post_date LIKE "'.$post_date.'"';
		} else {
			$conditions[] = 'post_date = "'.$post_date.'"';
		}
	}
	if ($use_prefixes) {
		foreach ($prefixes as $prefix) {
			$conditionsOR[] = 'guid LIKE "'.$prefix.'%"';
		}
		$conditions[] = '('.implode(' OR ', $conditionsOR).')';
	}
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE %s'
		.' ORDER BY id ASC',
		$wpdb->prefix.'posts',
		implode(' AND ', $conditions));
	$image = wpdb_get_row($query);
	return $image;
}

function getLocalImages($count = null, $where = array(), $processing_conditions = true) {
	global $wpdb, $prefixes;
	$conditions = array(
		'post_type = "attachment"',
		'post_mime_type LIKE "image/%"'
	);
	if ($processing_conditions) {
		$conditions[] = 'guid NOT LIKE "%#no-files#%"';
		$conditionsOR = array();
		foreach ($prefixes as $prefix) {
			$conditionsOR[] = 'guid LIKE "'.$prefix.'%"';
		}
		$conditions[] = '('.implode(' OR ', $conditionsOR).')';
	}
	if (is_array($where)) foreach ($where as $colkey => $colvalue) {
		if (substr($colvalue, 0, 1) == '%'
			|| substr($colvalue, -1) == '%')
		{
			$conditions[] = $colkey.' LIKE "'.$colvalue.'"';
		} else {
			$conditions[] = $colkey.' = "'.$colvalue.'"';
		}
	}
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE %s'
		.' ORDER BY id ASC'
		.' %s', // limit
		$wpdb->prefix.'posts',
		implode(' AND ', $conditions),
		!is_null($count) ? 'LIMIT '.$count : '');
	$images = wpdb_get_results($query);
	return $images;
}

function updateGUID($post_id, $new_guid) {
	global $wpdb;
	$mime_type  = fileMimeType($new_guid);
	$query = sprintf('UPDATE %s'
		.' SET guid = "%s", post_mime_type = "%s"'
		.' WHERE ID = %d',
		$wpdb->prefix.'posts',
		$wpdb->_real_escape($new_guid),
		$wpdb->_real_escape($mime_type),
		$post_id);
	return $wpdb->query($query);
}

function updatePostContent($post_id, $new_content) {
	global $wpdb;
	$query = sprintf('UPDATE %s'
		.' SET post_content = "%s"'
		.' WHERE ID = %d',
		$wpdb->prefix.'posts',
		$wpdb->_real_escape($new_content),
		$post_id);
	return $wpdb->query($query);
}

function getPostMeta($post_id) {
	global $wpdb;
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE post_id = %d',
		$wpdb->prefix.'postmeta',
		intval($post_id));
	$meta = wpdb_get_results($query);
	$ret = array();
	if (is_array($meta)) foreach ($meta as $key => $m) {
		$meta_key = $m['meta_key'];
		$meta_value = $m['meta_value'];
		if (is_serialized($meta_value)) {
			$meta_value = unserialize($meta_value);
		}
		$ret[$meta_key] = $meta_value;
	}
	return $ret;
}

function extractFilesFromMeta($meta) {
	$files = array();
	if (!is_array($meta)) {
		return $files;
	}
	if (isset($meta['file'])) {
		$files[] = $meta['file'];
	}
	$dir_prefix = rtrim(dirname($meta['file']), '/').'/';
	if (isset($meta['sizes']) && is_array($meta['sizes'])) {
		foreach ($meta['sizes'] as $size => $sizeinfo) {
			$files[] = $dir_prefix.$sizeinfo['file'];
		}
	}
	return $files;
}

function printWordpressAttachmentMetadata($metadata) {
	if (!is_array($metadata)) {
		writeline('    - metadata is not array', COLOR_RED);
		return false;
	}
	if (isset($metadata['file'])) {
		write('    - main file: ');
		write($metadata['file'], COLOR_BRIGHT_WHITE);
		write(' with size width = ');
		write($metadata['width'], COLOR_BRIGHT_BLUE);
		write(' and height = ');
		writeline($metadata['height'], COLOR_BRIGHT_BLUE);
	} else {
		writeline('    - FAIL: main file is not defined for this image', COLOR_BRIGHT_MAGENTA);
	}
	if (isset($metadata['sizes'])
		&& count($metadata['sizes']) > 0)
	{
		foreach ($metadata['sizes'] as $size_type => $sizes) {
			write('    - file for '.$size_type.': ');
			write($sizes['file'], COLOR_BRIGHT_WHITE);
			write(' with size width = ');
			write($sizes['width'], COLOR_BRIGHT_BLUE);
			write(' and height = ');
			writeline($sizes['height'], COLOR_BRIGHT_BLUE);
		}
	} else {
		writeline('    - FAIL: re-sized is not defined for this image', COLOR_BRIGHT_MAGENTA);
	}
	return true;
}

function checkMetadataExists($post_id) {
	global $wpdb;
	$query = sprintf('SELECT *'
					.' FROM %s'
					.' WHERE post_id = %d',
					$wpdb->prefix.'postmeta',
					intval($post_id));
	$data = wpdb_get_results($query);
	if (!is_array($data)) {
		return 0; // missing
	}
	$by_keys = array();
	foreach ($data as $row) {
		$by_keys[$row['meta_key']] = $row;
	}
	if (!isset($by_keys['_wp_attached_file'])
		|| !isset($by_keys['_wp_attachment_metadata']))
	{
		return -1; // corrupted
	}
	if ($by_keys['_wp_attachment_metadata']['meta_value'] == 'a:0:{}'
		|| strlen($by_keys['_wp_attachment_metadata']['meta_value']) < 10)
	{
		return -1; // corrupted
	}
	return 1; // OK
}

function updateAttachmentFile($post_id, $file_to_cdn_url) {
	global $wpdb;
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE post_id = %d'
		.' AND meta_key = "_wp_attached_file"',
		$wpdb->prefix.'postmeta',
		intval($post_id));
	$attachment_file = wpdb_get_row($query);
	$meta_value = $file_to_cdn_url[$attachment_file['meta_value']];
	$query = sprintf('UPDATE %s'
		.' SET meta_value = "%s"'
		.' WHERE post_id = %d'
		.' AND meta_key = "_wp_attached_file"',
		$wpdb->prefix.'postmeta',
		$wpdb->_real_escape($meta_value),
		intval($post_id));
	return $wpdb->query($query);
}

function updateAttachmentMetadata($post_id, $file_to_cdn_url) {
	global $wpdb;
	$query = sprintf('SELECT *'
		.' FROM %s'
		.' WHERE post_id = %d'
		.' AND meta_key = "_wp_attachment_metadata"',
		$wpdb->prefix.'postmeta',
		intval($post_id));
	$attachment_data = wpdb_get_row($query);
	if (empty($attachment_data)) {
		return false;
	}
	$meta_value = unserialize($attachment_data['meta_value']);
	$meta_value['file'] = $file_to_cdn_url[$meta_value['file']];
	$meta_value['mime-type'] = fileMimeType($meta_value['file']);
	if (isset($meta_value['sizes']) && is_array($meta_value['sizes'])) {
		foreach ($meta_value['sizes'] as $size => $sizeinfo) {
			$meta_value['sizes'][$size]['file'] = $file_to_cdn_url[$sizeinfo['file']];
		}
	}
	$meta_value = serialize($meta_value);
	$query = sprintf('UPDATE %s'
		.' SET meta_value = "%s"'
		.' WHERE post_id = %d'
		.' AND meta_key = "_wp_attachment_metadata"',
		$wpdb->prefix.'postmeta',
		$wpdb->_real_escape($meta_value),
		intval($post_id));
	return $wpdb->query($query);
}

function getPostWithImages($files) {
	global $wpdb;
	$ret = array();
	if (is_array($files)) foreach ($files as $key => $file) {
		if (strlen(trim($file)) <= 0) {
			continue;
		}
		$query = sprintf('SELECT *'
			.' FROM %s'
			.' WHERE post_content LIKE "%%%s%%"', // "%%<img src=\\"%s%%"
			$wpdb->prefix.'posts',
			$wpdb->_real_escape($file));
		$ret[$file] = wpdb_get_results($query);
	}

	return $ret;
}

$_id_min = null;
function getPostIdMin() {
	global $wpdb, $_id_min;
	if (is_null($_id_min)) {
		$query = sprintf('SELECT MIN(ID) AS id_min'
			.' FROM %s',
			$wpdb->prefix.'posts');
		$row = wpdb_get_row($query);
		$_id_min = $row['id_min'];
	}
	return $_id_min;
}

function getPostIdMinByDate($from_date) {
	global $wpdb;
	$query = sprintf('SELECT MIN(ID) AS id_min'
		.' FROM %s'
		.' WHERE post_modified_gmt >= "%s"'
		.' AND post_type != "attachment"',
		$wpdb->prefix.'posts',
		date('Y-m-d', strtotime($from_date)));
	$row = wpdb_get_row($query);
	$id_min = $row['id_min'];
	if (is_null($id_min)
		|| strlen($id_min) <= 0)
	{
		$id_min = getPostIdMin();
	}
	return $id_min;
}

$_id_max = null;
function getPostIdMax() {
	global $wpdb, $_id_max;
	if (is_null($_id_max)) {
		$query = sprintf('SELECT MAX(ID) AS id_max'
			.' FROM %s',
			$wpdb->prefix.'posts');
		$row = wpdb_get_row($query);
		$_id_max = $row['id_max'];
	}
	return $_id_max;
}

function getPostWithImages2($files) {
	global $wpdb, $run_in_cron;
	$ret = array();
	$id_min = getPostIdMin();
	$id_max = getPostIdMax();
	$id_step = 100000;
	$cnt_step = count($files) * max((($id_max - $id_min) / $id_step), 1);
	$pct_step = 100 / $cnt_step;
	$pct = 0;
	if (!$run_in_cron) {
		write(sprintf('%6.2f', $pct).' %');
	}
	if (is_array($files)) foreach ($files as $key => $file) {
		if (strlen(trim($file)) <= 0) {
			continue;
		}
		$ret[$file] = array();
		for ($id = $id_min; $id <= $id_max; $id += $id_step) {
			$query = sprintf('SELECT *'
				.' FROM %s'
				.' WHERE ID >= %d'
				.' AND ID < %d'
				.' AND post_content LIKE "%%%s%%"', // "%%<img src=\\"%s%%"
				$wpdb->prefix.'posts',
				$id,
				$id + $id_step,
				$wpdb->_real_escape($file));
			$partial = wpdb_get_results($query);
			$ret[$file] = array_merge($ret[$file], $partial);
			$pct += $pct_step;
			if (!$run_in_cron) {
				write(str_repeat(chr(8), 8));
				write(sprintf('%6.2f', $pct).' %');
			}
		}
	}
	if (!$run_in_cron) {
		// clear
		write(str_repeat(chr(8), 8));
		write(str_repeat(' ', 8));
		write(str_repeat(chr(8), 8));
	}
	return $ret;
}

function getPostWithImages3($files, $from_date = null) {
	global $wpdb, $run_in_cron;
	$ret = array();
	$id_min = is_null($from_date) ? getPostIdMin() : getPostIdMinByDate($from_date);
	write(' (ID min = ');
	write($id_min, COLOR_BRIGHT_BLUE);
	write(') ');
	$id_max = getPostIdMax();
	$id_step = 100000;
	$cnt_step = max((($id_max - $id_min) / $id_step), 1);
	$pct_step = 100 / $cnt_step;
	$pct = 0;
	if (!$run_in_cron) {
		write(sprintf('%6.2f', $pct).' %');
	}
	$files_optimized = array();
	if (is_array($files)) foreach ($files as $key => $file) {
		$file = preg_replace('/^https:/', '', $file);
		$file = preg_replace('/^http:/', '', $file);
		if (!in_array($file, $files_optimized)) {
			$files_optimized[] = $file;
		}
	}
	$cond_post_content = array();
	if (is_array($files_optimized)) foreach ($files_optimized as $key => $file) {
		if (strlen(trim($file)) <= 0) {
			continue;
		}
		$cond_post_content[] = sprintf('post_content LIKE "%%%s%%"', $wpdb->_real_escape($file));
	}
	$founded = array();
	if (count($cond_post_content) > 0) {
		for ($id = $id_min; $id <= $id_max; $id += $id_step) {
			$query = sprintf('SELECT *'
				.' FROM %s'
				.' WHERE ID >= %d'
				.' AND ID < %d'
				.' AND (%s)',
				$wpdb->prefix.'posts',
				$id,
				$id + $id_step,
				implode(' OR ', $cond_post_content));
			$partial = wpdb_get_results($query);
			$founded = array_merge($founded, $partial);
			$pct += $pct_step;
			if (!$run_in_cron) {
				write(str_repeat(chr(8), 8));
				write(sprintf('%6.2f', $pct).' %');
			}
		}
	}
	if (is_array($files)) foreach ($files as $key => $file) {
		$ret[$file] = array();
		foreach ($founded as $key => $post) {
			if (strpos($post['post_content'], $file) !== false) {
				$ret[$file][] = $post;
			}
		}
	}
	if (!$run_in_cron) {
		// clear
		write(str_repeat(chr(8), 8));
		write(str_repeat(' ', 8));
		write(str_repeat(chr(8), 8));
	}
	return $ret;
}

function getPostWithImages3Fork($files, $from_date = null) {
	global $fork_max_processes, $run_in_cron;
	$ret = array();
	$id_min = is_null($from_date) ? getPostIdMin() : getPostIdMinByDate($from_date);
	$id_max = getPostIdMax();
	$id_step = 100000;
	$cnt_step = max((($id_max - $id_min) / $id_step), 1);
	$pct_step = 100 / $cnt_step;
	$pct = 0;
	write(sprintf('%6.2f', $pct).' %');
	$pids = array();
	$pids_processed = array();
	$id = $id_min;
	if (!is_dir(__DIR__.'/forks')) {
		mkdir(__DIR__.'/forks', 0777, true);
	}
	clearDirectory(__DIR__.'/forks');
	while ($id < $id_max || count($pids) > 0) {
		if ($id < $id_max) {
			$pid = pcntl_fork();
			if ($pid == -1) {
				die('PCNTL_FORK() could not fork');
			} else if ($pid) {
				// we are the parent
				$pids[] = $pid;
				$id += $id_step;
			} else { // == 0; we are the child
				$partial = getPostWithImagesBetweenID($files, $id, $id + $id_step);
				//print_r($partial);
				file_put_contents(__DIR__.'/forks/pid-'.getmypid().'.txt', json_encode($partial));
				exit(0);
				break;
			}
			if (count($pids) < $fork_max_processes) continue; // fork next child
		}

		$return_pid = pcntl_wait($status); //Protect against Zombie children
		$pids_processed[] = $return_pid;
		array_remove($pids, $return_pid);
		$pct += $pct_step;
		if (!$run_in_cron) {
			write(str_repeat(chr(8), 8));
			write(sprintf('%6.2f', $pct).' %');
		}
	}
	// pozbieranie vysledkov
	$founded = array();
	foreach ($pids_processed as $pid) {
		$partial = json_decode(file_get_contents(__DIR__.'/forks/pid-'.$pid.'.txt'), true);
		$founded = array_merge($founded, $partial);
	}
	if (is_array($files)) foreach ($files as $key => $file) {
		$ret[$file] = array();
		foreach ($founded as $key => $post) {
			if (strpos($post['post_content'], $file) !== false) {
				$ret[$file][] = $post;
			}
		}
	}
	clearDirectory(__DIR__.'/forks');
	if (!$run_in_cron) {
		// clear
		write(str_repeat(chr(8), 8));
		write(str_repeat(' ', 8));
		write(str_repeat(chr(8), 8));
	}
	return $ret;
}

function getPostWithImagesBetweenID($files, $id_min, $id_max) {
	global $wpdb;
	$files_optimized = array();
	if (is_array($files)) foreach ($files as $key => $file) {
		$file = preg_replace('/^https:/', '', $file);
		$file = preg_replace('/^http:/', '', $file);
		if (!in_array($file, $files_optimized)) {
			$files_optimized[] = $file;
		}
	}
	$cond_post_content = array();
	if (is_array($files_optimized)) foreach ($files_optimized as $key => $file) {
		if (strlen(trim($file)) <= 0) {
			continue;
		}
		$cond_post_content[] = sprintf('post_content LIKE "%%%s%%"', $wpdb->_real_escape($file));
	}
	if (count($cond_post_content) <= 0) {
		return array();
	}
	$query = sprintf('SELECT *'
			.' FROM %s'
			.' WHERE ID >= %d'
			.' AND ID < %d'
			.' AND (%s)',
			$wpdb->prefix.'posts',
			$id_min,
			$id_max,
			implode(' OR ', $cond_post_content));
	$partial = wpdb_get_results($query);
	return $partial;
}


function getPostWithImages4($files, $from_date) {
	global $wpdb, $run_in_cron;
	$ret = array();
	if (is_null($from_date)
		|| empty($from_date))
	{
		return $ret;
	}
	$from_date_ts = strtotime($from_date);
	if ($from_date_ts === false
		|| $from_date_ts <= 0)
	{
		return $ret;
	}
	$now_ts = time();
	write(' (From date = ');
	write($from_date, COLOR_BRIGHT_BLUE);
	write(') ');
	$one_step = 28 * 24 * 3600; // 28 days
	$cnt_step = ceil(($now_ts - $from_date_ts) / $one_step);
	$pct_step = 100 / $cnt_step;
	$pct = 0;
	if (!$run_in_cron) {
		write(sprintf('%6.2f', $pct).' %');
	}
	$files_optimized = array();
	if (is_array($files)) foreach ($files as $key => $file) {
		$file = preg_replace('/^https:/', '', $file);
		$file = preg_replace('/^http:/', '', $file);
		if (!in_array($file, $files_optimized)) {
			$files_optimized[] = $file;
		}
	}
	$cond_post_content = array();
	if (is_array($files_optimized)) foreach ($files_optimized as $key => $file) {
		if (strlen(trim($file)) <= 0) {
			continue;
		}
		$cond_post_content[] = sprintf('post_content LIKE "%%%s%%"', $wpdb->_real_escape($file));
	}
	$founded = array();
	if (count($cond_post_content) > 0) {
		for ($ts = $from_date_ts; $ts <= $now_ts; $ts += $one_step) {
			$query = sprintf('SELECT *'
				.' FROM %s'
				.' WHERE post_modified_gmt >= "%s"'
				.' AND post_modified_gmt < "%s"'
				.' AND (%s)',
				$wpdb->prefix.'posts',
				date('Y-m-d H:i:s', $ts),
				date('Y-m-d H:i:s', $ts + $one_step),
				implode(' OR ', $cond_post_content));
			$partial = wpdb_get_results($query);
			$founded = array_merge($founded, $partial);
			$pct += $pct_step;
			if (!$run_in_cron) {
				write(str_repeat(chr(8), 8));
				write(sprintf('%6.2f', $pct).' %');
			}
		}
	}
	if (is_array($files)) foreach ($files as $key => $file) {
		$ret[$file] = array();
		foreach ($founded as $key => $post) {
			if (strpos($post['post_content'], $file) !== false) {
				$ret[$file][] = $post;
			}
		}
	}
	if (!$run_in_cron) {
		// clear
		write(str_repeat(chr(8), 8));
		write(str_repeat(' ', 8));
		write(str_repeat(chr(8), 8));
	}
	return $ret;
}

function cache_flush() {
	write('Remove all cache items ... ');
	$suc = wp_cache_flush();
	if ($suc) {
		writeline('OK', COLOR_GREEN);
	} else {
		writeline('FAILED', COLOR_RED);
	}
	return $suc;
}

/**
 * wrapping for Wordpress Database Handler
 */
function get_platon_die_handler() {
	return 'platon_die_handler';
}

function platon_die_handler( $message, $title="", $args = array() ) {
	global $wpdb;
	$message = strip_tags($message);
	writeline('FAILED: '.$message, COLOR_RED);
	if (stristr($title, 'datab') != false) {
		$wpdb->last_error = 'CONNECTION LOST';
		$max_try = PLATON_CDN_DB_RECONNECT_TRY;
		while (!$wpdb->check_connection(false) && $max_try-- > 0) {
			writeline('WPDB FAILED: [ '.date('Y-m-d H:i:s').' ] reconnect to DB failed, sleep 10 seconds and try again', COLOR_RED);
			sleep(PLATON_CDN_DB_RECONNECT_DELAY);
			writeline('WPDB: [ '.date('Y-m-d H:i:s').' ] try reconnect now', COLOR_RED);
		}
		writeline('WPDB: [ '.date('Y-m-d H:i:s').' ] OK, connected to DB, continue', COLOR_GREEN);
		$wpdb->last_error = 'RECONNECTED';
		$wpdb->query('SET SESSION net_read_timeout = 3600');
		$wpdb->query('SET SESSION net_write_timeout = 3600');
		return true;
	}
	die();
}

function wpdb_get($type, $query) {
	global $wpdb;
	$try_again = true;
	while ($try_again) {
		$try_again = false;
		$error_message = null;
		try {
			switch ($type) {
				case 'ROW': $ret = $wpdb->get_row($query, ARRAY_A); break;
				case 'COL': $ret = $wpdb->get_col($query); break;
				case 'RESULTS': $ret = $wpdb->get_results($query, ARRAY_A); break;
				default: return false;
			}
		} catch (Exception $e) {
			$error_message = $e->getMessage();
		}
		if (is_null($error_message)
			&& !empty($wpdb->last_error))
		{
			$error_message = $wpdb->last_error;
		}
		if (!is_null($error_message)
			&& (stristr($error_message, 'gone away') !== false
				|| stristr($error_message, 'RECONNECTED') !== false ))
		{
			writeline('WPDB FAILED: [ '.date('Y-m-d H:i:s').' ] '.$error_message, COLOR_RED);
			$try_again = true;
			while (!$wpdb->check_connection(false)) {
				writeline('WPDB FAILED: [ '.date('Y-m-d H:i:s').' ] reconnect to DB failed, sleep 10 seconds and try again', COLOR_RED);
				sleep(10);
			}
			$wpdb->query('SET SESSION net_read_timeout = 3600');
			$wpdb->query('SET SESSION net_write_timeout = 3600');
		}
	}
	return $ret;
}

function wpdb_get_row($query) {
	return wpdb_get('ROW', $query);
}

function wpdb_get_col($query) {
	return wpdb_get('COL', $query);
}

function wpdb_get_results($query) {
	return wpdb_get('RESULTS', $query);
}

/**
 * Tests
 */
function runTests() {
	write('Long query about 5 seconds ... ', COLOR_WHITE);
	writeTimeBegin();
	$result = wpdb_get_results('select benchmark(20000000, md5("when will it end?"))');
	write('Result: ', COLOR_GRAY);
	var_dump($result);
	if (is_array($result)
		&& count($result) > 0)
	{
		write('OK ', COLOR_BRIGHT_GREEN);
	} else {
		write('FAILED ', COLOR_BRIGHT_RED);
	}
	writeTimeEnd();

	write('Long query about 30 seconds ... ', COLOR_WHITE);
	writeTimeBegin();
	$result = wpdb_get_results('select benchmark(140000000, md5("when will it end?"))');
	write('Result: ', COLOR_GRAY);
	var_dump($result);
	if (is_array($result)
		&& count($result) > 0)
	{
		write('OK ', COLOR_BRIGHT_GREEN);
	} else {
		write('FAILED ', COLOR_BRIGHT_RED);
	}
	writeTimeEnd();

	writeline('Done.', COLOR_GREEN);
}

?>
