<?php
/**
* Plugin Name: Cluster Commands
* Plugin URI: https://platon.org/
* Description: Frontend for cluster commands.
* Version: 1.6
* Author: Platon Technologies
* Author URI: https://platon.org/
* Update URI: https://wordpress.platon.sk/cluster-commands/
* License: GPL2
*/

define('PLATON_CLUSTERCOMMANDS_DB_VERSION', 'PLATON_CLUSTERCOMMANDS_DB_VERSION');

class Platon_ClusterCommands {

	public static $version = '1.6';

	public function __construct() {
		$this->checkDB();
		$this->doActions();
	}

	public static function menu() {
		add_submenu_page(
			'tools.php',
			'Cluster Commands',
			'Cluster Commands',
			'moderate_comments', //'manage_options',
			plugin_dir_path(__FILE__).'admin/view.php'
		);
	}

	public function onActivate() {
		global $wpdb;
		$query = 'SHOW TABLES LIKE "'.$wpdb->prefix.'platon_cluster_commands"';
		$table = $wpdb->get_row($query, ARRAY_A);
		if (is_null($table)) {
			$query = 'CREATE TABLE `'.$wpdb->prefix.'platon_cluster_commands` (
					`command_id` INT(11) PRIMARY KEY AUTO_INCREMENT,
					`command_type` ENUM("REQUEST", "ANSWER") NOT NULL,
					`command_dt` DATETIME NOT NULL,
					`request_id` INT(11) DEFAULT NULL,
					`user_id` INT(11) DEFAULT NULL,
					`server_name` VARCHAR(32) DEFAULT NULL,
					`command` VARCHAR(32) DEFAULT NULL,
					`result` VARCHAR(255) DEFAULT NULL
				)';
			$wpdb->query($query);
		}
	}

	public function checkDB() {
		global $wpdb;
		$db_version = get_option(PLATON_CLUSTERCOMMANDS_DB_VERSION, 0);
		if ($db_version <= 0) {
			$query = 'DESCRIBE  "'.$wpdb->prefix.'platon_cluster_commands" "result"';
			$col_check = $wpdb->get_row($query, ARRAY_A); echo 'col_check='; print_r($col_check);
			if (is_null($col_check)) {
				$fix_query = 'ALTER TABLE `'.$wpdb->prefix.'platon_cluster_commands` ADD `result` VARCHAR(255) DEFAULT NULL AFTER `command`';
				$wpdb->query($fix_query);
			}
			update_option(PLATON_CLUSTERCOMMANDS_DB_VERSION, 1);
		}
	}

	public function onDeactivate() {
	}

	// https://rudrastyh.com/wordpress/self-hosted-plugin-update.html
	public function pluginsAPI($res, $action, $args) {
		// do nothing if this is not about getting plugin information
		if( 'plugin_information' !== $action ) {
			return $res;
		}

		// do nothing if it is not our plugin
		if( plugin_basename( __DIR__ ) !== $args->slug ) {
			return $res;
		}

		// info.json is the file with the actual plugin information on your server
		$remote = wp_remote_get(
			'https://wordpress.platon.sk/cluster-commands/info.json',
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json'
				)
			)
		);

		// do nothing if we don't get the correct response from the server
		if (is_wp_error( $remote )
			|| 200 !== wp_remote_retrieve_response_code( $remote )
			|| empty( wp_remote_retrieve_body( $remote )))
		{
			return $res;
		}

		$remote = json_decode( wp_remote_retrieve_body( $remote ) );

		$res = new stdClass();
		$res->name = $remote->name;
		$res->slug = $remote->slug;
		$res->author = $remote->author;
		$res->author_profile = $remote->author_profile;
		$res->version = $remote->version;
		$res->tested = $remote->tested;
		$res->requires = $remote->requires;
		$res->requires_php = $remote->requires_php;
		$res->download_link = $remote->download_url;
		$res->trunk = $remote->download_url;
		$res->last_updated = $remote->last_updated;
		$res->sections = array(
			'description' => $remote->sections->description,
			'installation' => $remote->sections->installation,
			'changelog' => $remote->sections->changelog
			// you can add your custom sections (tabs) here
		);
		// in case you want the screenshots tab, use the following HTML format for its content:
		// <ol><li><a href="IMG_URL" target="_blank"><img src="IMG_URL" alt="CAPTION" /></a><p>CAPTION</p></li></ol>
		if( ! empty( $remote->sections->screenshots ) ) {
			$res->sections[ 'screenshots' ] = $remote->sections->screenshots;
		}

		$res->banners = array(
			'low' => $remote->banners->low,
			'high' => $remote->banners->high
		);

		return $res;
	}

	public function updatePlugin( $transient ) {
		//print_r($transient);
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$remote = wp_remote_get(
			'https://wordpress.platon.sk/cluster-commands/info.json',
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json'
				)
			)
		);

		if (is_wp_error( $remote )
			|| 200 !== wp_remote_retrieve_response_code( $remote )
			|| empty( wp_remote_retrieve_body( $remote )))
		{
			return $transient;
		}

		$remote = json_decode( wp_remote_retrieve_body( $remote ) );

		// your installed plugin version should be on the line below! You can obtain it dynamically of course
		if(
			$remote
			&& version_compare( Platon_ClusterCommands::$version, $remote->version, '<' )
			&& version_compare( $remote->requires, get_bloginfo( 'version' ), '<=' )
			&& version_compare( $remote->requires_php, PHP_VERSION, '<=' )
		) {

			$res = new stdClass();
			$res->slug = $remote->slug;
			$res->plugin = plugin_basename( __FILE__ ); // it could be just YOUR_PLUGIN_SLUG.php if your plugin doesn't have its own directory
			$res->new_version = $remote->version;
			$res->tested = $remote->tested;
			$res->package = $remote->download_url;
			$transient->response[ $res->plugin ] = $res;

			//$transient->checked[$res->plugin] = $remote->version;
		}

		return $transient;

	}

	public function commands() {
		$commands = array_merge(
			$this->commandsPhpFpm(),
			array(
			'restart_nginx'          => 'Restart Nginx',
			'restart_redis_session'  => 'Restart Session Redis',
			'restart_redis_cache'    => 'Restart Cache Redis',
			'restart_varnish'        => 'Restart Varnish',
			'flush_redis_session'    => 'Flush Session Redis',
			'flush_redis_cache'      => 'Flush Cache Redis'
			)
		);
		if (defined('CLUSTER_COMMANDS_ALLOWED')
			&& is_array(CLUSTER_COMMANDS_ALLOWED))
		{
			$commands_new = array();
			foreach ($commands as $command_key => $command_label) {
				if (!$this->in_array_stars($command_key, CLUSTER_COMMANDS_ALLOWED)) {
					continue;
				}
				$commands_new[$command_key] = $command_label;
			}
			$commands = $commands_new;
		}
		if (defined('CLUSTER_COMMANDS_DISABLED')
			&& is_array(CLUSTER_COMMANDS_DISABLED))
		{
			foreach ($commands as $command_key => $command_label) {
				if ($this->in_array_stars($command_key, CLUSTER_COMMANDS_DISABLED)) {
					unset($commands[$command_key]);
				}
			}
		}
		return $commands;
	}

	private function commandsPhpFpm() {
		$all = glob('/etc/init.d/php*-fpm');
		if (!is_array($all)) return array();
		$ret = array();
		foreach ($all as $init_script) {
			$base_name = basename($init_script);
			preg_match('/php(.*)-fpm/', $base_name, $matches);
			$version = is_array($matches) && isset($matches[1]) ? $matches[1] : '';
			$ret['restart_'.$base_name] = 'Restart PHP FPM '.$version;
		}
		return $ret;
	}

	private function stars_match($stars, $str) {
		$begin = substr($stars, 0, 1) == '*';
		$end = substr($stars, -1) == '*';
		$stars_clear = trim($stars, '*');
		if ($begin && $end) { // sufix
			if (stristr($str, $stars_clear) !== false) {
				return true;
			}
		} else if ($begin && !$end) { // postfix
			if (substr($str, -1 * strlen($stars_clear)) == $stars_clear) {
				return true;
			}
		} else if (!$begin && $end) { // prefix
			if (substr($str, 0, strlen($stars_clear)) == $stars_clear) {
				return true;
			}
		} else { // full-match
			if ($str == $stars) {
				return true;
			}
		}
		return false;
	}

	private function stars_in_array($item, $arr) {
		$begin = substr($item, 0, 1) == '*';
		$end = substr($item, -1) == '*';
		$item_clear = trim($item, '*');
		if (!is_array($arr)) return false;
		foreach ($arr as $a) {
			if ($begin && $end) { // sufix
				if (stristr($a, $item_clear) !== false) {
					return true;
				}
			} else if ($begin && !$end) { // postfix
				if (substr($a, -1 * strlen($item_clear)) == $item_clear) {
					return true;
				}
			} else if (!$begin && $end) { // prefix
				if (substr($a, 0, strlen($item_clear)) == $item_clear) {
					return true;
				}
			} else { // full-match
				if ($a == $item) {
					return true;
				}
			}
		}
		return false;
	}

	private function in_array_stars($item, $arr) {
		if (!is_array($arr)) return false;
		foreach ($arr as $a) {
			$begin = substr($a, 0, 1) == '*';
			$end = substr($a, -1) == '*';
			$a_clear = trim($a, '*');
			if ($begin && $end) { // sufix
				if (stristr($item, $a_clear) !== false) {
					return true;
				}
			} else if ($begin && !$end) { // postfix
				if (substr($item, -1 * strlen($a_clear)) == $a_clear) {
					return true;
				}
			} else if (!$begin && $end) { // prefix
				if (substr($item, 0, strlen($a_clear)) == $a_clear) {
					return true;
				}
			} else { // full-match
				if ($a == $item) {
					return true;
				}
			}
		}
		return false;
	}

	public function doActions() {
		if (isset($_REQUEST['save'])) {
			$this->createRequest($_REQUEST['command']);
		}
        if (isset($_REQUEST['shortcut'])) {
			$this->createRequest($_REQUEST['command']);
		}
	}

	public function createRequest($command) {
		global $wpdb;
		if (strlen($command) <= 0) return false;
		$query = $wpdb->prepare('INSERT INTO '.$wpdb->prefix.'platon_cluster_commands'
						.' (command_type, command_dt, user_id, command)'
						.' VALUES ("REQUEST", NOW(), %d, %s)',
						get_current_user_id(),
						$command);
		return $wpdb->query($query);
	}

	public function getRequests($reverse = false, $limit = null) {
		global $wpdb;
		$query = 'SELECT command_id FROM '.$wpdb->prefix.'platon_cluster_commands WHERE command_type = "REQUEST" ORDER BY command_id '.($reverse ? 'DESC' : 'ASC');
		if (!is_null($limit)) {
			$query .= ' LIMIT '.$limit;
		}
		$command_ids = $wpdb->get_col($query);
		$query = 'SELECT * FROM '.$wpdb->prefix.'platon_cluster_commands'
				.' WHERE command_id IN ('.implode(', ', $command_ids).')'
				.' OR request_id IN ('.implode(', ', $command_ids).')'
				.' ORDER BY command_id';
		$all = $wpdb->get_results($query, ARRAY_A);
		$ret = array();
		if (is_array($all)) foreach ($all as $one) {
			if ($one['command_type'] == 'REQUEST') {
				$ret[$one['command_id']] = array(
					'request' => $one,
					'answers' => array()
				);
				continue;
			}
			if ($one['command_type'] == 'ANSWER') {
				$ret[$one['request_id']]['answers'][] = $one;
				continue;
			}
		}
		if ($reverse) krsort($ret);
		return $ret;
	}

	public function getRequestsCount() {
		global $wpdb;
		$query = 'SELECT COUNT(*) AS count FROM '.$wpdb->prefix.'platon_cluster_commands WHERE command_type = "REQUEST"';
		return $wpdb->get_var($query);
	}

}

function platon_cluster_commands_init() {
	$GLOBALS['platon_cluster_commands'] = new Platon_ClusterCommands();
}

add_action('admin_init', 'platon_cluster_commands_init' );
add_action('admin_menu', array('Platon_ClusterCommands', 'menu'));
register_activation_hook(__FILE__, array('Platon_ClusterCommands', 'onActivate'));
register_deactivation_hook( __FILE__, array('Platon_ClusterCommands','onDeactivate') );
add_filter( 'plugins_api', array('Platon_ClusterCommands', 'pluginsAPI'), 20, 3);
add_filter( 'site_transient_update_plugins', array('Platon_ClusterCommands', 'updatePlugin') );



?>
